#include <internal/nested_tags_checker.h>
#include <internal/config.h>
#include <string>

#include <butil/Expat.h>

namespace msg_body {

static const std::string divTag = "div";

class XmlHandler {
public:

    XmlHandler(const Configuration& config) noexcept
        : depthLimit(config.divTagDepthLimit),
          depth(0),
          limitExceeded(false)
    {
    }

    XmlHandler(const XmlHandler&) = delete;

    void onStartElement(const char* name) noexcept {
        if (!limitExceeded && divTag == name) {
            ++depth;
            if (depth > depthLimit) {
                limitExceeded = true;
            }
        }
    }

    void onEndElement(const char* name) noexcept {
        if (divTag == name && depth > 0) {
            --depth;
        }
    }

    bool isTagsDepthLimitExceeded() const noexcept {
        return limitExceeded;
    }

private:
    const unsigned depthLimit;
    unsigned depth;
    bool limitExceeded;
};

inline XmlHandler* getHandler(void* data) noexcept {
    return static_cast<XmlHandler*>(data);
}

NestedTagsChecker::NestedTagsChecker(const Configuration& config,
                                     const LogPtr& logger) noexcept
    : logger(logger),
      config(config)
{
}

NestedTagsCheckerResult NestedTagsChecker::check(const std::string& content) {
    if (content.empty()) {
        return NestedTagsCheckerResult::Ok;
    }

    Expat xmlParser("UTF-8");
    XmlHandler handler(config);

    XML_StartElementHandler startHandler = [](void* data, const char* name, const char**) noexcept {
        getHandler(data)->onStartElement(name);
    };

    XML_EndElementHandler endHandler = [](void* data, const char* name) noexcept {
        getHandler(data)->onEndElement(name);
    };

    xmlParser.setUserData(&handler);
    xmlParser.setStartElementHandler(startHandler);
    xmlParser.setEndElementHandler(endHandler);

    const bool parseResult = xmlParser.parse(content);

    if (handler.isTagsDepthLimitExceeded()) {
        return NestedTagsCheckerResult::TagsDepthLimitExceeded;
    }

    if (parseResult != XML_STATUS_OK) {
        using namespace std::literals;
        MBODY_LOG_DEBUG(logger, log::where_name="tags_check", log::message="html parse failed: "s + xmlParser.error() + ", in line: " +
            std::to_string(xmlParser.errorLine()) + ", at column: "  + std::to_string(xmlParser.errorColumn()));
        return NestedTagsCheckerResult::ParseError;
    }

    return NestedTagsCheckerResult::Ok;
}

}
