#include <internal/zip_iterator.h>
#include <mail_getter/UTFizer.h>
#include <yamail/data/deserialization/yajl.h>
#include <internal/passbook_parse.h>

namespace msg_body {

using boost::property_tree::ptree;

static const std::string passbookDictionary = "pass.json";

bool isPassbookDictionary(const ZipEntry& entry) {
    return entry.header()->getFileName() == passbookDictionary;
}

std::string extractPassDict(const std::string& zippedPackage) {
    std::istringstream inputStream(zippedPackage);
    zipios::ZipInputStream zipStream(inputStream);

    ZipIterator zipIter = std::find_if(ZipIterator(zipStream), ZipIterator(), &isPassbookDictionary);
    if (zipIter == ZipIterator()) {
        throw PassbookPackageException("Can't find " + passbookDictionary + " in package");
    }
    return zipIter->data();
}

PassbookPackage parsePassbookDict(const std::string& passDict) {
    PassbookPackage res;
    try {
        res.dict = yamail::data::deserialization::json::toPtree(passDict);
    } catch (const std::bad_alloc&) {
        throw;
    } catch (const std::exception& e) {
        throw PassbookPackageException("failed to parse " + passbookDictionary + ": " + e.what());
    }
    return res;
}

static std::string utfizeIfUtf16(const Recognizer::Wrapper& recognizer, const std::string& str) {
    std::string res = str;
    UTFizer utfizer(recognizer);
    if (utfizer.recognizeEncoding(str) == Recognizer::CODES_UTF16) {
        utfizer.utfize("utf-16", res);
    }
    return res;
}

PassbookPackage parsePassbookPackage(const Recognizer::Wrapper& recognizer, const std::string& data) {
    std::string passDict = extractPassDict(data);
    passDict = utfizeIfUtf16(recognizer, passDict);
    return parsePassbookDict(passDict);
}

}
