#include <internal/sanitize_transform.h>
#include <internal/pa_log.h>

#include <yplatform/encoding/url_encode.h>

#include <algorithm>

namespace {

const std::string redirectorPrefix = "http://mail.yandex.ru/r?url=";

} // namespace

using namespace mail_getter;

namespace msg_body {

class SanitizeTransformer {
public:
    SanitizeTransformer(const InlineSpoofer& inlineSpoofer, const SanitizeTransformParams& params)
        : inlineSpoofer_(inlineSpoofer)
        , params_(params) {}

    SanitizeTransformResult operator() (SanitizerParsedResponse& sanitizeData) const {
        SanitizeTransformResult res;
        res.content.swap(sanitizeData.html);
        safeTransformContentByMarkupInReverseRange(res, sanitizeData.markup);
        return res;
    }

private:
    const InlineSpoofer& inlineSpoofer_;
    const SanitizeTransformParams& params_;

    void safeTransformContentByMarkupInReverseRange(SanitizeTransformResult &res, const SanitizerMarkup& markup) const {
        std::for_each(markup.rbegin(), markup.rend(), [&](const auto& entry) {
            safeTransformContentByMarkupEntry(res, entry);
        });
    }

    void safeTransformContentByMarkupEntry(SanitizeTransformResult &res, const SanitizerMarkupEntry& entry) const {
        switch (entry.type) {
            case SanitizerMarkupType_Cid:
                if (params_.cidParts) {
                    transformCid(res, entry, *params_.cidParts);
                } 
                break;

            case SanitizerMarkupType_UnsafeLink:
                transformUnsafeLink(res, entry);
                break;

            default:
                break;
        }
    }

    void transformCid(SanitizeTransformResult& res, const SanitizerMarkupEntry& entry, const CidParts& cidParts) const {
        const std::string& cid = getCid(res.content, entry);
        auto it = cidParts.find(cid);
        if (it != cidParts.end()) {
            const CidPartInfo& cidInfo = it->second;
            const std::string& url = inlineSpoofer_.getUrl(cidInfo);
            replaceCidtoUrl(res.content, url, entry);
            res.cids.insert(cid);
        } else {
            removeAttr(res.content, entry);
        }
    }

    void transformUnsafeLink(SanitizeTransformResult& res, const SanitizerMarkupEntry& entry) const {
        const auto& url = std::string(res.content.substr(entry.position.dataStart, entry.position.dataLength));
        const auto& encodedUrl = yplatform::url_encode<std::string>(url);
        const auto& wrappedLink = redirectorPrefix + encodedUrl;

        res.content.replace(entry.position.dataStart, entry.position.dataLength, wrappedLink);
    }

    std::string getCid(const std::string &content, const SanitizerMarkupEntry& entry) const {
        std::string cid;
        if (content.substr(entry.position.dataStart, 4) == "cid:") {
            cid = content.substr(entry.position.dataStart + 4, entry.position.dataLength - 4);
        }
        return cid;
    }
    void replaceCidtoUrl(std::string &content, const std::string &url,
            const SanitizerMarkupEntry& entry) const {
        content.replace(entry.position.dataStart, entry.position.dataLength, url);
    }
    void removeAttr(std::string &content, const SanitizerMarkupEntry& entry) const {
        content.erase(entry.position.attributeStart, entry.position.attributeLength);
    }
};

SanitizeTransformResult sanitizeTransform(const InlineSpoofer& inlineSpoofer,
    mail_getter::SanitizerParsedResponse& sanitizeData, const SanitizeTransformParams& params) {
    const PaLog paLog(__FUNCTION__);
    SanitizeTransformResult res = (SanitizeTransformer(inlineSpoofer, params))(sanitizeData);
    paLog.write();
    return res;
}

} // namespace msg_body
