#include <string>
#include <map>
#include <set>
#include <boost/tokenizer.hpp>
#include <boost/algorithm/string.hpp>
#include <boost/range/algorithm/copy.hpp>

using namespace std::literals;

namespace msg_body {
namespace sanitizer {

const std::set<std::string_view> tags = {"br/", "br", "hr", "/h1", "/h2", "/h3", "/h4", "/h5", "/h6", "/p", "/li", "/tr", "/p"};

const std::map<std::string_view, std::string_view> skipTags = {
    {"script", "/script"},
    {"style", "/style"}
};


// Boost.Tokenizer operates via std::string representation of tokens
// since it is old and there was no std::string_view at that time.
// But it has customization points, thus we can still use the library
// a little bit customized and optimized for our purposes.
struct tagSeparator {
    using token = std::string_view;

    template <typename Iter>
    bool operator() (Iter& next, Iter end, token& out) const {
        if (next == end) {
            return false;
        }

        const auto isTagSeparator = [&](auto ch) { return ch == '<' || ch == '>';};

        Iter i = isTagSeparator(*next) ? std::next(next) : std::find_if(next, end, isTagSeparator);
        out = token(std::addressof(*next), std::distance(next, i));
        next = i;
        return true;
    }

    void reset() const {}
};

auto tokenizer(const std::string& in) {
    using token = tagSeparator::token;
    return boost::make_iterator_range(
        boost::make_token_iterator<token>(in.begin(), in.end(), tagSeparator{}),
        boost::make_token_iterator<token>(in.end(), in.end(), tagSeparator{})
    );
}

// The call could be used for inplace changes, but it is more safe to make a copy.
// The function could be rewritten as a functional object if desired.
template <typename Out>
Out sanitize(const std::string& str, Out out) {
    enum class ParserState {outsideTag, insideTag};

    auto skipTag = skipTags.end();

    const auto insideSkipTag = [&] { return skipTag != skipTags.end(); };

    const auto stopSkipTag = [&] { skipTag = skipTags.end(); };

    const auto startSkipTag = [&](auto i) { skipTag = i; };

    const auto isClosingSkipTag = [&](auto& tag) { return skipTag->second == tag; };

    const auto getTag = [](auto& token) {
        static const auto nonTagNameChars = "/ \t\r\n";
        const auto start = token.find_first_not_of(nonTagNameChars);
        const auto end = token.find_first_of(nonTagNameChars, start);
        const auto isClosing = token.find_first_of("/") < token.find_first_not_of(nonTagNameChars);
        // The only copy occurs here since case insensitive comparision of string_view is painful
        // But you could eliminate it if choose to write own implementation of iless.
        std::string tag;
        if (isClosing) {
            tag += '/';
        }
        boost::algorithm::to_lower_copy(std::back_inserter(tag), token.substr(start, end-start));
        return tag;
    };

    const auto isClosing = [](auto& tag) { return !tag.empty() && tag.front() == '/';};

    // state and tagToken could be tied together within some context structure
    // if needed.
    ParserState state = ParserState::outsideTag;

    std::string_view tagToken;

    // we have two obvious states here, so it is easy to find
    // a part of code to customize or optimize the neccessary step
    // or function
    const auto handleToken = [&](auto& token, Out out) {
        switch (state) {
        case ParserState::outsideTag:
            if (token == "<") {
                state = ParserState::insideTag;
            } else if (!insideSkipTag()) {
                out = boost::copy(token, out);
            }
            break;

        case ParserState::insideTag:
            if (token == ">") {
                state = ParserState::outsideTag;
                const auto tag = getTag(tagToken);
                if (tags.count(tag)) {
                    // Here you can still use a mapping, but it seems like it not a necessary
                    // thing for now
                    out = boost::copy("<br>"sv, out);
                } else if (insideSkipTag()) {
                    if (isClosingSkipTag(tag)) {
                        stopSkipTag();
                    }
                } else if (auto i = skipTags.find(tag); i != skipTags.end() ) {
                    startSkipTag(i);
                } else if (isClosing(tag)) {
                    *out++ = ' ';
                }
            } else {
                tagToken = token;
            }
            break;
        }
        return out;
    };

    for (auto& token : tokenizer(str)) {
        out = handleToken(token, out);
    }
    return out;
}

void sanitizeByTykva(std::string& str) {
    str.erase(sanitize(str, str.begin()), str.end());
}

} // namespace sanitizer
} // namespace msg_body
