#include <internal/server/config_change_handler.h>
#include <internal/server/message_handler.h>
#include <internal/server/server.h>
#include <internal/server/headers_handler.h>
#include <internal/server/message_source_handler.h>
#include <internal/server/message_text_handler.h>
#include <internal/config.h>
#include <internal/exceptions.h>
#include <internal/server/request_context.h>
#include <internal/server/reflection.h>
#include <internal/logger.h>

#include <ymod_webserver/server.h>
#include <ymod_webserver/response.h>

#include <yplatform/find.h>

#include <yamail/data/serialization/yajl.h>

#include <internal/pa_log.h>

namespace msg_body {

namespace {

template <ymod_webserver::methods::http_method method, class Handler>
auto allowOnly(Handler&& handler) {
    return [handler = std::forward<Handler>(handler)]
        (const ymod_webserver::http::stream_ptr& stream) {
            if (stream->request()->method == method) {
                return handler(stream);
            }
            stream->set_code(ymod_webserver::codes::method_not_allowed);
        };
}

template <class Handler>
auto allowOnlyGet(Handler&& handler) {
    return allowOnly<ymod_webserver::methods::mth_get>(std::forward<Handler>(handler));
}

void rejectByTvmTicket(RequestContext& request, const std::string& method) {
    request.rejectByTvmTicket();
    MBODY_LOG_ERROR(request.logger(), log::where_name=method, log::message="bad tvm ticket");
}

template <class Handler, class TvmGuardPtr, class TvmGuardLogger>
auto wrap(const std::string& name, Handler&& handler, TvmGuardPtr guard, TvmGuardLogger guardLogger, std::size_t coroutineStackSize) {
    return [name, handler = std::forward<Handler>(handler), guard = std::move(guard), guardLogger = std::move(guardLogger), coroutineStackSize]
        (const ymod_webserver::http::stream_ptr& stream) {
            spawn([name, handler = std::move(handler), guard = std::move(guard), guardLogger = std::move(guardLogger),
                    stream = std::move(stream)] (YieldCtx yield) {
                using yamail::data::serialization::toJson;
                RequestContext context(stream);
                try {
                    const auto tvm = guard->check(context);
                    MBODY_LOG_INFO(guardLogger, log::request_id=context.requestId(), log::where_name=name,
                        log::tvm_guard::response=tvm);

                    if (tvm.action == tvm_guard::Action::reject) {
                        rejectByTvmTicket(context, name);
                        return;
                    }
                    const auto result = handler(context, yield);

                    const auto body = toJson(result).str();
                    context.responseSuccess(body);
                } catch (const boost::coroutines::detail::forced_unwind&) {
                    throw;
                } catch (const ParamsException& e) {
                    const std::string msg="failed to parse params";
                    reflection::Response response {"invalid request", msg + ": " + e.what()};
                    context.responseInvalidRequest(toJson(response));
                    MBODY_LOG_ERROR(context.logger(), log::where_name=name, log::message=msg, log::exception=e);
                } catch (const mail_getter::UnknownMid& e) {
                    const std::string msg="exception: error in forming message";
                    reflection::Response response {"invalid request", msg + ": " + e.what()};
                    context.responseInvalidRequest(toJson(response));
                    MBODY_LOG_ERROR(context.logger(), log::where_name=name, log::message=msg, log::exception=e);
                } catch (const mail_getter::MessageNotFound& e) {
                    const std::string msg="exception: error in forming message";
                    reflection::Response response {"invalid request", msg + ": " + e.what()};
                    context.responseInvalidRequest(toJson(response));
                    MBODY_LOG_ERROR(context.logger(), log::where_name=name, log::message=msg, log::exception=e);
                } catch (const std::bad_alloc& e) {
                    const std::string msg="bad_alloc, will abort";
                    reflection::Response response {"internal error", msg + ": " + e.what()};
                    context.responseInternalError(toJson(response));
                    MBODY_LOG_ERROR(context.logger(), log::where_name=name, log::message=msg, log::exception=e);
                } catch (const boost::system::system_error& e) {
                    const std::string msg = "system_error";
                    reflection::Response response{"internal error", msg + ": " + e.what()};
                    context.responseInternalError(toJson(response));
                    MBODY_LOG_ERROR(context.logger(), log::where_name=name, log::exception=e);
                } catch (const std::exception& e) {
                    const std::string msg="exception";
                    reflection::Response response {"internal error", msg + ": " + e.what()};
                    context.responseInternalError(toJson(response));
                    MBODY_LOG_ERROR(context.logger(), log::where_name=name, log::exception=e);
                } catch (...) {
                    const std::string msg="unknown error";
                    reflection::Response response {"internal error", msg};
                    context.responseInternalError(toJson(response));
                    MBODY_LOG_ERROR(context.logger(), log::where_name=name, log::message=msg);
                }
            }, coroutineStackSize);

        };
}

} // namespace

void bindServerHandles(const Configuration& config) {
    const auto server = yplatform::find<ymod_webserver::server, std::shared_ptr>("http_server");
    server->bind("", {"/ping"}, allowOnlyGet([] (const ymod_webserver::http::stream_ptr& stream) {
        stream->set_code(ymod_webserver::codes::ok);
        stream->set_content_type("text", "plain");
        stream->result_body("pong");
    }));
    const auto bind = [&] (const std::string& name, auto&& handler) {
        server->bind("", {"/" + name}, allowOnlyGet(wrap(name, std::move(handler),
            config.tvmGuard, config.tvmGuardLogger, config.coroutineStackSize)));
    };

    bind("message", MessageHandler(config));
    bind("headers", HeadersHandler(config));
    bind("message_source", MessageSourceHandler(config));
    bind("v1/message/text", MessageTextHandler(config));
}

void bindConfigChangeHandler(Configuration& config) {
    const auto server = yplatform::find<ymod_webserver::server, std::shared_ptr>("http_server");
    constexpr auto name = "pumpkin";
    server->bind("", {"/pumpkin/enable", "/pumpkin/disable"}, allowOnly<ymod_webserver::methods::mth_post>(wrap(name, ConfigChangeHandler(config),
        config.tvmGuard, config.tvmGuardLogger, config.coroutineStackSize)));
}

} // namespace msg_body
