#include <boost/regex.hpp>
#include <boost/algorithm/string.hpp>
#include <boost/algorithm/string/iter_find.hpp>
#include <butil/butil.h>
#include <butil/StrUtils/Encoding.h>
#include <butil/StrUtils/StrUtils.h>
#include <mail_getter/UTFizer.h>
#include <internal/transformer_attributes.h>
#include <internal/transformers/narod_transformer.h>

namespace msg_body {

static const boost::regex narodRegex(
    "href=\"(?:(.*[&\?]hash=([^\"&]*))?([^\"]*))\""
    "(?: data-preview=\"([^\"]*)\")?"
    "(?: data-hash=\"([^\"]*)\")?"
    "[^>]*>(.*) [(]([^)]*)[)]"
);
NarodTransformer::NarodTransformer(const TransformerAttributes& attrs,
                                   const IContentTypeDetector& contentTypeDetector,
                                   const AliasClassList& mimeTypesList,
                                   const Recognizer::Wrapper& recognizer)
    : attrs_(attrs)
    , contentTypeDetector_(contentTypeDetector)
    , mimeTypesList_(mimeTypesList)
    , recognizer(recognizer)
{}

void decodeText(std::string& content, const TransformerAttributes& attrs, MetaPart& hs,
                const Recognizer::Wrapper& recognizer) {
    Encoding::decode(hs.encoding(), content);
    const std::string& charset = attrs.useCharset.empty() ? hs.charset() : attrs.useCharset;
    utfizeString(recognizer, content, charset);
    TStrUtils::replaceBadSymbols(content);
}

bool isNarodAttachFolder(const std::string& sizeDescription) {
    const std::string trimmed = boost::algorithm::trim_copy(sizeDescription);
    return !isdigit(trimmed[0]);
}

TransformNarodAttachResultPtr NarodTransformer::transformAttach(const std::string& html, const std::string& fakeHid) const {
    boost::smatch match;
    if (!boost::regex_search(html, match, narodRegex)) {
        return TransformNarodAttachResultPtr();
    }
    TransformNarodAttachResultPtr res = std::make_unique<TransformNarodAttachResult>();
    res->name = match[6].str();
    res->partClassInfo = determinePartClass(contentTypeDetector_,
        mimeTypesList_, res->name, "application", "octet-stream");
    res->fakeHid = fakeHid;
    
    if (match[5].matched) {
        res->hash = match[5].str();
    }
    if (match[1].matched) {
        res->url = match[1].str();
        if (res->hash.empty()) {
            res->hash = match[2].str();
        }
    } else {
        res->url = match[3].str();
    }
    res->sizeDescription = match[7].str();
    res->isFolder = isNarodAttachFolder(res->sizeDescription);

    if (match[4].matched) {
        res->preview = decode_url(match[4].str());
    }
    return res;
}

NarodTransformerResult NarodTransformer::apply(MessagePart& part) const {
    NarodTransformerResult res;
    decodeText(part.content, attrs_, part.headerStruct, recognizer);
    std::vector<std::string> attaches;
    boost::iter_split(attaches, part.content, boost::first_finder("</a>"));
    res.reserve(attaches.size());

    for (size_t i = 0; i < attaches.size(); ++i) {
        const std::string fakeHid = part.hid + "." + boost::lexical_cast<std::string>(i+1);
        TransformNarodAttachResultPtr transformRes = transformAttach(attaches[i], fakeHid);
        if (transformRes.get()) {
            res.push_back(*transformRes);
        }
    }
    return res;
}

}
