#include <internal/transformer_attributes.h>
#include <internal/message_tree.h>
#include <internal/message_context.h>
#include <internal/passbook_parse.h>
#include <internal/transformers/transformers.h>
#include <internal/transformers/routines.h>

namespace msg_body {

Transformers::Transformers(const Configuration& config,
        TransformerAttributes& attrs,
        MessageContext& messageContext,
        LogPtr logger,
        const IContentTypeDetector& contentTypeDetector,
        const AliasClassList& aliasClassList,
        unsigned trimThreshold,
        VdirectPtr vdirect,
        const Sanitizer& sanitizer,
        const InlineSpoofer& inlineSpoofer,
        const Recognizer::Wrapper& recognizer,
        const TextAsyncFactExtractor& asyncFactExtractor)
    : attrs_(attrs)
    , messageContext_(messageContext)
    , logger_(logger)
    , pkpassTransformer_(attrs_.mid, attrs_.from, logger, recognizer)
    , messageTransformer_(contentTypeDetector, aliasClassList, recognizer)
    , narodTransformer_(attrs_, contentTypeDetector, aliasClassList, recognizer)
    , binaryTransformer_(attrs_,
        messageContext_,
        logger,
        contentTypeDetector,
        aliasClassList)
    , textTransformer_(config,
        attrs_,
        messageContext_,
        logger,
        contentTypeDetector,
        aliasClassList,
        trimThreshold,
        vdirect,
        sanitizer,
        inlineSpoofer,
        recognizer,
        asyncFactExtractor)
{}

BinaryTransformerResultPtr Transformers::applyBinaryTransformer(MessagePart& part) const {
    BinaryTransformerResult res = binaryTransformer_.apply(part);
    return BinaryTransformerResultPtr(new BinaryTransformerResult(res));
}

CalendarTransformerResultPtr Transformers::applyCalendarTransformer(MessagePart& part) const {
    CalendarTransformerResult res = calendarTransformer_.apply(part);
    return CalendarTransformerResultPtr(new CalendarTransformerResult(res));
}

MessageTransformerResultPtr Transformers::applyMessageTransformer(MessagePart& part) const {
    MessageTransformerResult res = messageTransformer_.apply(part);
    return MessageTransformerResultPtr(new MessageTransformerResult(res));
}

NarodTransformerResultPtr Transformers::applyNarodTransformer(MessagePart& part) const {
    NarodTransformerResult res = narodTransformer_.apply(part);
    return NarodTransformerResultPtr(new NarodTransformerResult(res));
}

TextTransformerResultPtr Transformers::applyTextTransformer(MessagePart& part) const {
    TextTransformerResult res = textTransformer_.apply(part);
    return TextTransformerResultPtr(new TextTransformerResult(res));
}

PassbookPackagePtr Transformers::applyPkpassTransformer(MessagePart& part) const {
    PassbookPackage passbook = pkpassTransformer_.apply(part);
    return PassbookPackagePtr(new PassbookPackage(passbook));
}

TransformersResultPtr Transformers::apply(MessagePart& part, const std::string& transformerType) const {
    try {
        TransformersResultPtr res(new TransformersResult);
        if (transformerType == "binary") {
            res->binaryTransformerResult = applyBinaryTransformer(part);
            MBODY_LOG_DEBUG(logger_, log::where_name="transformers", log::message="attach: inline=" + std::to_string(res->binaryTransformerResult->isInline) + ", estimate_size=" + std::to_string(estimateSize(part.headerStruct.length(), part.headerStruct.encoding())));
        } else if (transformerType == "calendar") {
            res->calendarTransformerResult = applyCalendarTransformer(part);
            MBODY_LOG_DEBUG(logger_, log::where_name="transformers", log::message="calendar_part: size=" + std::to_string(res->calendarTransformerResult->length));
        } else if (transformerType == "message") {
            res->messageTransformerResult = applyMessageTransformer(part);
            MBODY_LOG_DEBUG(logger_, log::where_name="transformers", log::message="attached_msg: estimate_size=" + std::to_string(estimateSize(part.headerStruct.length(), part.headerStruct.encoding())));
        } else if (transformerType == "narod") {
            res->narodTransformerResult = applyNarodTransformer(part);
            MBODY_LOG_DEBUG(logger_, log::where_name="transformers", log::message="has narod attach");
        } else if (transformerType == "signed") {
            // nothing
        } else if (transformerType == "text") {
            res->textTransformerResult = applyTextTransformer(part);
            if (!res->textTransformerResult->content.empty()) {
                MBODY_LOG_DEBUG(logger_, log::message="text_body: size=" + std::to_string(res->textTransformerResult->content.size()));
            }
        } else if (transformerType == "pkpass") {
            res->pkpassTransformerResult = applyPkpassTransformer(part);
        } else {
            throw TransformersException("Unknown transformer type " + transformerType);
        }
        return res;
    } catch (const TransformersException& e) {
        MBODY_LOG_ERROR(logger_, log::where_name="transformers", log::message="failed to apply", log::exception=e);
        return TransformersResultPtr();
    } catch (const std::bad_alloc&) {
        throw;
    } catch (const std::exception& e) {
        throw MessagePartException(e.what(), part.hid);
    }
}

}
