/* Alexey Kuzyakov <alexbft@yandex-team.ru> 11.08.2008
 *
 * UUENCODE parser
 *
 */

#include <string.h>
#include <string>
#include <map>
#include <ctype.h>
#include <boost/lexical_cast.hpp>
#include <internal/uudecode.h>

#ifdef __clang__
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wsign-conversion"
#endif

using namespace msg_body;
using namespace uudecode;

typedef std::map <std::string, std::string> STR_MAP;
extern STR_MAP mimetype;

const std::string MIME_UNKNOWN = "unknown";
const char base64_abc[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";
int base64map[256];
bool is_base64map_init = false;

typedef unsigned char uchar;

void rtrim(std::string &s)
{
    int x = static_cast<int>(s.length()) - 1;
    while (x >= 0 && (s[x] == ' ' || s[x] == '\r' || s[x] == '\n' || s[x] == '\t')) --x;
    if (x < 0) s = "";
    else if (x < static_cast<int>(s.length()) - 1) s = s.substr(0, x);
}

bool read_prefix(const char *&s, const char *prefix)
{
    const char *cur = s;
    while (*prefix && *cur == *prefix)
    {
        ++cur;
        ++prefix;
    }
    if (*prefix == 0)
    {
        s = cur;
        return true;
    }
    return false;
}

bool read_to_eoln(const char *&s, std::string &result)
{
    const char *cur = s;
    while (*cur && *cur != '\n') ++cur;
    if (*cur == 0) return false;
    result.assign(s, cur - s);
    rtrim(result);
    s = cur + 1;
    return true;
}

inline int decode_tr(uchar c)
{
    if (c < 32) return -1;
    else return (c - 32) & 63;
}

inline int decode_64(uchar c)
{
    return base64map[c];
}

const char * ChunkWrapper::create_from(const char *begin)
{
    if (!read_prefix(begin, "begin")) return 0;
    if (read_prefix(begin, "-base64")) return create_from_base64(begin);
    else return create_from_traditional(begin);
}

bool ChunkWrapper::read_header(const char *&begin)
{
    //space
    if (*begin != ' ') return false;
    ++begin;

    //reading unix rights (e.g. '644')
    for (int i = 0; i < 3; ++i, ++begin)
    {
        if (*begin <= '0' || *begin >= '7') return false;
    }

    //space
    if (*begin != ' ') return false;
    ++begin;

    //reading file name (e.g. 'filename.ext')
    if (!read_to_eoln(begin, chunk_.file_name)) return false;

    return true;
}

bool ChunkWrapper::read_line_traditional(const char *&s, bool &is_end, int &last_line_length)
{
    std::string buf;
    if (!read_to_eoln(s, buf)) return false;
    if (buf == "end") {
        if (last_line_length == 0) {
            is_end = true;
            return true;
        }
        else return false;
    }
    if (buf.length() < 1) return false;
    int len = decode_tr(buf[0]);
    buf = buf.substr(1);
    if (len < 0 || len > 45 || static_cast<size_t>(len) > buf.length() / 4 * 3) return false;
    int n = (len + 2) / 3;
    std::string decoded = "";
    for (int i = 0; i < n; ++i)
    {
        int ii = i * 4;
        int a3 = decode_tr(buf[ii]);
        int a2 = decode_tr(buf[ii + 1]);
        int a1 = decode_tr(buf[ii + 2]);
        int a0 = decode_tr(buf[ii + 3]);
        if (a3 < 0 || a2 < 0 || a1 < 0 || a0 < 0) return false;
        int res = (a3 << 18) | (a2 << 12) | (a1 << 6) | a0;
        int r0 = res & 0xFF; res = res >> 8;
        int r1 = res & 0xFF; res = res >> 8;
        int r2 = res & 0xFF;
        decoded.push_back(static_cast<uchar>(r2));
        decoded.push_back(static_cast<uchar>(r1));
        decoded.push_back(static_cast<uchar>(r0));
    }
    chunk_.m_content.append(decoded.substr(0, len));
    last_line_length = len;
    return true;
}

void base64map_init()
{
    memset(base64map, -1, sizeof(base64map));
    for (int i = 0; i < 64; ++i)
        base64map[uchar(base64_abc[i])] = i;
    base64map[uchar('=')] = 0;
    is_base64map_init = true;
}

bool ChunkWrapper::read_line_base64(const char *&s, bool &is_end)
{
    if (!is_base64map_init) base64map_init();
    std::string buf;
    if (!read_to_eoln(s, buf)) return false;
    if (buf == "====") {
        is_end = true;
        return true;
    }
    if (buf.length() % 4 != 0) return false;
    int n = static_cast<int>(buf.length() / 4);
    for (int i = 0; i < n; ++i)
    {
        int ii = i * 4;
        int a3 = decode_64(buf[ii]);
        int a2 = decode_64(buf[ii + 1]);
        int a1 = decode_64(buf[ii + 2]);
        int a0 = decode_64(buf[ii + 3]);
        if (a3 < 0 || a2 < 0 || a1 < 0 || a0 < 0) return false;
        int res = (a3 << 18) | (a2 << 12) | (a1 << 6) | a0;
        int r0 = res & 0xFF; res = res >> 8;
        int r1 = res & 0xFF; res = res >> 8;
        int r2 = res & 0xFF;
        chunk_.m_content.push_back(static_cast<uchar>(r2));
        if (buf[ii + 2] != '=') {
            chunk_.m_content.push_back(static_cast<uchar>(r1));
            if (buf[ii + 3] != '=') chunk_.m_content.push_back(static_cast<uchar>(r0));
        }
    }
    return true;
}

const char * ChunkWrapper::create_from_traditional(const char *begin)
{
    if (!read_header(begin)) return 0;
    chunk_.m_content = "";
    bool is_end = false;
    int last_line_length = -1;
    while (!is_end)
    {
        if (!read_line_traditional(begin, is_end, last_line_length)) return 0;
    }
    return begin;
}

const char * ChunkWrapper::create_from_base64(const char *begin)
{
    if (!read_header(begin)) return 0;
    chunk_.m_content = "";
    bool is_end = false;
    while (!is_end)
    {
        if (!read_line_base64(begin, is_end)) return 0;
    }
    return begin;
}

std::string str_lowercase(std::string s)
{
    for (size_t i = 0; i < s.length(); ++i)
    {
        s[i] = static_cast<char>(tolower(s[i]));
    }
    return s;
}

std::string get_mime_type(const std::string &fn)
{
    size_t x = fn.rfind('.');
    if (x != std::string::npos)
    {
        std::string ext = str_lowercase(fn.substr(x + 1));
        STR_MAP::iterator it = mimetype.find(ext);
        if (it != mimetype.end())
            return it->second;
        else
            return MIME_UNKNOWN;
    }
    else
        return MIME_UNKNOWN;
}

void ChunkWrapper::setMimeType() {
    chunk_.mime_type = get_mime_type(chunk_.file_name);
}

inline bool uuencode_precheck(const char *chunk_begin)
{
    return *chunk_begin == 'b';
}

void uudecode::GetUuencodeXMLRepresentation(std::string &pattern, ChunkWrappers& chunks) {
    chunks.clear();

    const char *chunk_begin = pattern.c_str();

    std::string result = "";
    result.reserve(pattern.size());

    ChunkWrapper chunk;
    size_t chunk_num = 0;
    while (*chunk_begin)
    {
        const char *chunk_end;
        if (uuencode_precheck(chunk_begin) && (chunk_end = chunk.create_from(chunk_begin)))
        {
            //new uuencoded chunk found
            ++chunk_num;
            chunks.push_back(ChunkWrapper());
            chunk.setMimeType();
            chunks.back().swap(chunk);
            chunk_begin = chunk_end;
        }
        else
        {
            result.push_back(*chunk_begin);
            ++chunk_begin;
        }
    }

    if (chunk_num > 0) pattern = result;
}

#ifdef __clang__
#pragma clang diagnostic pop
#endif
