package ru.yandex.mail.micronaut.blackbox;

import javax.annotation.Nullable;

import io.micronaut.http.HttpRequest;
import io.micronaut.security.authentication.Authentication;
import lombok.AllArgsConstructor;
import lombok.val;
import org.reactivestreams.Publisher;
import reactor.core.publisher.Mono;

import static java.util.Objects.requireNonNullElse;

@AllArgsConstructor
public class BlackboxAuthenticationFetcher {
    private final String prefix;
    private final BlackboxTokenClient client;

    private Mono<BlackboxAuthentication> getAuthentication(String token, String userip) {
        return client.auth(token, userip)
            .map(BlackboxAuthentication::new)
            .filter(BlackboxAuthentication::isAuthenticated);
    }

    private @Nullable String getAuthToken(@Nullable String header) {
        if (header == null || !header.startsWith(prefix)) {
            return null;
        }

        return header.substring(prefix.length());
    }

    public Publisher<Authentication> fetchAuthentication(HttpRequest<?> request) {
        val headers = request.getHeaders();

        val token = getAuthToken(headers.get(io.micronaut.http.HttpHeaders.AUTHORIZATION));
        if (token == null) {
            return Mono.empty();
        }

        val userip = requireNonNullElse(headers.get(HttpHeaders.X_REAL_IP), "");
        return getAuthentication(token, userip)
            .cast(Authentication.class);
    }
}
