package ru.yandex.mail.micronaut.blackbox;

import java.util.Base64;

import javax.inject.Inject;
import javax.inject.Singleton;

import io.micronaut.context.annotation.Requires;
import io.micronaut.http.HttpRequest;
import io.micronaut.security.authentication.Authentication;
import io.micronaut.security.filters.AuthenticationFetcher;
import lombok.extern.slf4j.Slf4j;
import org.reactivestreams.Publisher;
import reactor.core.publisher.Mono;

@Slf4j
@Singleton
@Requires(property = "blackbox.basic")
@Requires(beans = BlackboxClient.class)
public class BlackboxBasicAuthenticationFetcher implements AuthenticationFetcher {
    @Inject
    private BlackboxClient client;
    private final BlackboxAuthenticationFetcher fetcher;

    public BlackboxBasicAuthenticationFetcher() {
        fetcher = new BlackboxAuthenticationFetcher("Basic ", this::getBlackboxResponse);
    }

    protected Mono<BlackboxResponse> getBlackboxResponse(String token, String userip) {
        final String[] decoded;
        try {
            decoded = new String(Base64.getDecoder().decode(token)).split(":");
        } catch (Exception e) {
            log.error("Error while decoding Authorization header for ip " + userip, e);
            return BlackboxResponse.failure();
        }

        if (decoded.length != 2) {
            log.error("An invalid token came from ip {} that does not conform to the standard Base64 representation of the login:password", userip);
            return BlackboxResponse.failure();
        }

        return client.login(decoded[0], userip, decoded[1]);
    }

    @Override
    public Publisher<Authentication> fetchAuthentication(HttpRequest<?> request) {
        return fetcher.fetchAuthentication(request);
    }
}
