package ru.yandex.mail.micronaut.common;

import lombok.AllArgsConstructor;
import lombok.experimental.UtilityClass;
import lombok.val;
import org.reactivestreams.Publisher;
import org.reactivestreams.Subscriber;
import org.reactivestreams.Subscription;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Operators;

import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.atomic.AtomicLongFieldUpdater;
import java.util.function.Supplier;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;

@UtilityClass
public class Async {
    private static final CompletableFuture<Void> VOID_FUTURE_DONE = completedFuture(null);

    public <T> CompletableFuture<T> asyncSafe(Supplier<CompletableFuture<T>> supp) {
        try {
            return supp.get();
        } catch (CompletionException e) {
            return failedFuture(e.getCause());
        } catch (Throwable e) {
            return failedFuture(e);
        }
    }

    public Throwable unwrap(Throwable exception) {
        return exception instanceof CompletionException ? exception.getCause() : exception;
    }

    public <T> CompletableFuture<T> done(T value) {
        return completedFuture(value);
    }

    public CompletableFuture<Void> done() {
        return VOID_FUTURE_DONE;
    }

    public CompletableFuture<Void> runIf(boolean condition, Supplier<CompletableFuture<Void>> action) {
        return condition ? action.get() : done();
    }

    @FunctionalInterface
    public interface PageFetcher<ID, T> {
        CompletableFuture<Page<ID, T>> fetch(Pageable<ID> pageable);
    }

    @AllArgsConstructor
    private static final class PagePublisher<ID, T> implements Publisher<List<T>> {
        private final int pageSize;
        private final PageFetcher<ID, T> fetcher;

        private static final class PageSubscription<ID, T> implements Subscription {
            private static final AtomicLongFieldUpdater<PageSubscription> REQUESTED_UPDATER =
                AtomicLongFieldUpdater.newUpdater(PageSubscription.class, "requested");

            private final PageFetcher<ID, T> fetcher;
            private final Subscriber<? super List<T>> subscriber;
            private Pageable<ID> current;
            private volatile long requested;
            private volatile boolean canceled;

            private PageSubscription(int pageSize, PageFetcher<ID, T> fetcher, Subscriber<? super List<T>> subscriber) {
                this.fetcher = fetcher;
                this.subscriber = subscriber;
                this.current = Pageable.first(pageSize);
                this.requested = 0;
                this.canceled = false;
            }

            private void fetch() {
                if (canceled) {
                    return;
                }

                fetcher.fetch(current)
                    .whenComplete((page, e) -> {
                        if (canceled) {
                            return;
                        }

                        if (e != null) {
                            subscriber.onError(e);
                        } else {
                            val elements = page.getElements();
                            if (!elements.isEmpty()) {
                                subscriber.onNext(elements);
                            }

                            if (page.getNextPageId().isEmpty()) {
                                subscriber.onComplete();
                                return;
                            }

                            val nextPageId = page.getNextPageId().get();
                            current = new Pageable<>(nextPageId, current.getPageSize());

                            val pending = Operators.produced(REQUESTED_UPDATER, this, 1);
                            if (pending > 0) {
                                fetch();
                            }
                        }
                    });
            }

            @Override
            public void request(long amount) {
                if (!Operators.validate(amount)) {
                    return;
                }

                var pending = Operators.addCap(REQUESTED_UPDATER, this, amount);
                if (pending > 0) {
                    return;
                }

                fetch();
            }

            @Override
            public void cancel() {
                canceled = true;
            }
        }

        @Override
        public void subscribe(Subscriber<? super List<T>> subscriber) {
            subscriber.onSubscribe(new PageSubscription<>(pageSize, fetcher, subscriber));
        }
    }

    public <ID, T> Flux<List<T>> fetchPagesRx(int pageSize, PageFetcher<ID, T> fetcher) {
        return Flux.from(new PagePublisher<>(pageSize, fetcher));
    }
}
