package ru.yandex.mail.micronaut.common;

import lombok.experimental.UtilityClass;
import one.util.streamex.StreamEx;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.stream.Collector;
import java.util.stream.Collectors;

import static java.util.function.Function.identity;
import static java.util.function.Predicate.not;

@UtilityClass
public class CerberusUtils {
    public static <T, U> List<U> mapToList(Collection<T> collection, Function<T, U> mapper) {
        return StreamEx.of(collection)
            .map(mapper)
            .toImmutableList();
    }

    public static <T, U> Set<U> mapToSet(Collection<T> collection, Function<T, U> mapper) {
        return StreamEx.of(collection)
            .map(mapper)
            .toImmutableSet();
    }

    public static <T, K, V> Map<K, V> mapToMap(Collection<T> collection, Function<T, K> keyMapper,
                                               Function<T, V> valueMapper) {
        return StreamEx.of(collection)
            .toMap(keyMapper, valueMapper);
    }

    public static <T, V> Map<T, V> mapToMapByKey(Collection<T> collection, Function<T, V> valueMapper) {
        return StreamEx.of(collection)
            .toMap(valueMapper);
    }

    public static <T, K> Map<K, T> mapToMapByValue(Collection<T> collection, Function<T, K> keyMapper) {
        return StreamEx.of(collection)
            .toMap(keyMapper, identity());
    }

    public static <T, U, R> List<R> foldToList(Collection<T> first, Collection<U> second, BiFunction<T, U, R> combiner) {
        if (first instanceof List && second instanceof List) {
            return StreamEx.zip((List<T>) first, (List<U>) second, combiner)
                .toImmutableList();
        } else {
            return StreamEx.of(first)
                .zipWith(second.stream())
                .map(pair -> combiner.apply(pair.getKey(), pair.getValue()))
                .toImmutableList();
        }
    }

    public static <T extends Enum<T>> Set<String> nameSetOf(Class<T> enumType) {
        return StreamEx.of(enumType.getEnumConstants())
            .map(Enum::name)
            .toImmutableSet();
    }

    /**
     * Search for the elements missing in search set
     * @param elements Set of elements to search
     * @param search Collection of elements where the search is done
     * @param collector result collector
     * @return List of missing elements
     */
    public static <T, A, R> R findMissing(Set<T> elements, Collection<T> search, Collector<T, A, R> collector) {
        if (search.isEmpty()) {
            return StreamEx.<T>of().collect(collector);
        }

        return StreamEx.of(search)
            .filter(not(elements::contains))
            .collect(collector);
    }

    public static <T> List<T> findMissing(Set<T> elements, Collection<T> search) {
        return findMissing(elements, search, Collectors.toList());
    }

    public static <T, U> List<T> findMissing(Set<T> elements, Collection<U> search, Function<U, T> mapper) {
        return findMissing(elements, StreamEx.of(search).map(mapper).toImmutableList());
    }
}
