package ru.yandex.mail.micronaut.common;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectReader;
import lombok.AllArgsConstructor;
import lombok.SneakyThrows;
import lombok.val;
import one.util.streamex.StreamEx;

import javax.inject.Inject;
import javax.inject.Singleton;

@Singleton
@AllArgsConstructor(onConstructor_= @Inject)
public class JsonMapper {
    private final ObjectMapper objectMapper;

    @SneakyThrows
    public <T> String toJson(T value) {
        if (value instanceof RawJsonString) {
            return ((RawJsonString) value).getValue();
        } else {
            return objectMapper.writeValueAsString(value);
        }
    }

    public <T> JsonNode toJsonNode(T value) {
        if (value instanceof RawJsonString) {
            return toJsonNode(((RawJsonString) value).getValue());
        } else {
            return objectMapper.valueToTree(value);
        }
    }

    @SneakyThrows
    @SuppressWarnings("unchecked")
    public <T> T fromJson(String json, Class<T> type) {
        if (type.equals(RawJsonString.class)) {
            return (T) new RawJsonString(json);
        } else {
            return objectMapper.readValue(json, type);
        }
    }

    @SneakyThrows
    @SuppressWarnings("unchecked")
    public <T> T fromJson(JsonNode node, Class<T> type) {
        if (type.equals(RawJsonString.class)) {
            return (T) new RawJsonString(node.toString());
        } else {
            return objectMapper.treeToValue(node, type);
        }
    }

    public <T> T fromJson(RawJsonString json, Class<T> type) {
        return fromJson(json.getValue(), type);
    }

    @SneakyThrows
    public JsonNode fromJson(String json) {
        return objectMapper.readTree(json);
    }

    @SneakyThrows
    public JsonNode fromJson(String json, DeserializationFeature... features) {
        val reader = StreamEx.of(features).foldLeft(objectMapper.reader(), ObjectReader::with);
        return reader.readTree(json);
    }
}
