package ru.yandex.mail.micronaut.common;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import com.fasterxml.jackson.databind.annotation.JsonNaming;
import io.micronaut.core.annotation.Introspected;
import lombok.AllArgsConstructor;
import lombok.Value;

import java.util.List;
import java.util.Optional;
import java.util.function.Function;

import static java.util.Collections.emptyList;
import static ru.yandex.mail.micronaut.common.CerberusUtils.mapToList;

@Value
@Introspected
@AllArgsConstructor(onConstructor_= @JsonCreator)
@JsonNaming(PropertyNamingStrategy.SnakeCaseStrategy.class)
public class Page<ID, T> {
    @SuppressWarnings("unchecked")
    private static final Page EMPTY = new Page(emptyList(), Optional.empty());

    List<T> elements;
    Optional<ID> nextPageId;

    public Page(List<T> elements) {
        this(elements, Optional.empty());
    }

    public boolean hasNext() {
        return nextPageId.isPresent();
    }

    @SuppressWarnings("unchecked")
    public static <ID, T> Page<ID, T> empty() {
        return EMPTY;
    }

    public <U> Page<ID, U> mapElements(Function<T, U> mapper) {
        return new Page<>(mapToList(elements, mapper), nextPageId);
    }

    public <ID2> Page<ID2, T> mapNextPageId(Function<ID, ID2> mapper) {
        return new Page<>(elements, nextPageId.map(mapper));
    }

    public <ID2, U> Page<ID2, U> map(Function<ID, ID2> idMapper, Function<T, U> elementMapper) {
        return new Page<>(mapToList(elements, elementMapper), nextPageId.map(idMapper));
    }
}
