package ru.yandex.mail.micronaut.common.context;

import io.micronaut.http.HttpRequest;
import io.micronaut.http.context.ServerRequestContext;
import lombok.val;
import org.apache.logging.log4j.ThreadContext;
import ru.yandex.mail.micronaut.common.AuthenticationInfo;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ThreadLocalRandom;

public class ContextManager {
    public static final String REQUEST_ID_HEADER = "X-Request-Id";

    private static final String MDC_REQUEST_ID_KEY = "rid";
    private static final String MDC_UID_KEY = "uid";
    private static final String MDC_CLIENT_NAME_KEY = "clientName";
    private static final String MDC_CLIENT_ID_KEY = "clientId";

    private static String generateRequestId() {
        return String.format("%016X", ThreadLocalRandom.current().nextLong());
    }

    private static String setRequestIdFrom(HttpRequest<?> request) {
        val id = request.getHeaders().get(REQUEST_ID_HEADER);
        val requestId = (id == null) ? generateRequestId() : id;
        request.setAttribute(REQUEST_ID_HEADER, requestId);
        ThreadContext.put(MDC_REQUEST_ID_KEY, requestId);
        return requestId;
    }

    private static void setAuthenticationInfoFrom(HttpRequest<?> request) {
        request.getUserPrincipal(AuthenticationInfo.class).ifPresent(info -> {
            info.userUid().ifPresent(uid -> ThreadContext.put(MDC_UID_KEY, String.valueOf(uid)));
            ThreadContext.put(MDC_CLIENT_NAME_KEY, info.clientName());
            ThreadContext.put(MDC_CLIENT_ID_KEY, String.valueOf(info.clientId()));
        });
    }

    static String setInformationFrom(HttpRequest<?> request) {
        setAuthenticationInfoFrom(request);
        return setRequestIdFrom(request);
    }

    /**
     * Returns request id from MDC.
     * Returns empty id if there is no request id within MDC.
     */
    public static String currentRequestId() {
        val result = ThreadContext.get(MDC_REQUEST_ID_KEY);
        return result == null ? "" : result;
    }

    /**
     * Returns authentication info of current request is processing in a controller.
     * Returns nothing outside of a controller scope.
     */
    public static Optional<AuthenticationInfo> currentRequestAuthenticationInfo() {
        return currentRequest()
            .flatMap(request -> request.getUserPrincipal(AuthenticationInfo.class));
    }

    /**
     * Returns current HTTP request is processing in a controller.
     * Returns nothing outside of a controller context.
     */
    public static <T> Optional<HttpRequest<T>> currentRequest() {
        return ServerRequestContext.currentRequest();
    }

    /**
     * Replaces current thread MDC context by the values from passed <code>context</code>
     */
    public static void switchMdcContext(Map<String, String> context) {
        clearMdcContext();
        if (!context.isEmpty()) {
            ThreadContext.putAll(context);
        }
    }

    public static void clearMdcContext() {
        ThreadContext.clearMap();
    }

    /**
     * Sets request id in current thread MDC
     */
    public static void setRequestId(String value) {
        ThreadContext.put(MDC_REQUEST_ID_KEY, value);
    }

    /**
     * Reset request id in current thread MDC
     */
    public static void resetRequestId() {
        ThreadContext.remove(MDC_REQUEST_ID_KEY);
    }

    public static void setMdcValue(String key, String value) {
        ThreadContext.put(key, value);
    }

    public static void setMdcValues(Map<String, String> values) {
        ThreadContext.putAll(values);
    }

    public static void resetMdcValue(String key) {
        ThreadContext.remove(key);
    }

    public static void resetMdcValues(String... keys) {
        ThreadContext.removeAll(List.of(keys));
    }

    public static void resetMdcValues(Iterable<String> keys) {
        ThreadContext.removeAll(keys);
    }

    /**
     * Sets random request id in current thread MDC
     * @return Generated request id
     */
    public static String setRandomRequestId() {
        val id = generateRequestId();
        setRequestId(id);
        return id;
    }
}
