package ru.yandex.mail.micronaut.http_logger.http_logger_log4j2;

import io.micronaut.http.HttpMethod;
import io.micronaut.http.HttpRequest;
import io.micronaut.http.HttpResponse;
import lombok.AccessLevel;
import lombok.experimental.FieldDefaults;
import lombok.val;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.util.StringBuilderFormattable;
import ru.yandex.mail.micronaut.http_logger.RequestLogger;
import ru.yandex.mail.micronaut.http_logger.config.LoggingOptions;
import ru.yandex.mail.micronaut.http_logger.message.MessageBuilder;
import ru.yandex.mail.micronaut.http_logger.message.RequestMessage;
import ru.yandex.mail.micronaut.http_logger.message.ResponseMessage;

@FieldDefaults(makeFinal = true, level = AccessLevel.PRIVATE)
final class Log4j2RequestLogger implements RequestLogger {
    @SuppressWarnings("NonConstantLogger")
    Logger log;
    LoggingOptions config;
    MessageBuilder messageBuilder;

    Log4j2RequestLogger(String loggerName, LoggingOptions config) {
        this.config = config;
        this.log = LogManager.getLogger(loggerName);
        this.messageBuilder = new MessageBuilder(config);
    }

    private StringBuilderFormattable toFormattable(RequestMessage message) {
        return builder -> messageBuilder.buildMessage(builder, message);
    }

    private StringBuilderFormattable toFormattable(ResponseMessage message) {
        return builder -> messageBuilder.buildMessage(builder, message);
    }

    @Override
    public <B> void logRequest(HttpRequest<B> request, String action) {
        if (config.isEnabled()) {
            val message = new RequestMessage(config, request, action);
            log.info(toFormattable(message));
        }
    }

    @Override
    public <B> void logResponse(HttpMethod requestMethod, String requestPath, HttpResponse<B> response) {
        if (config.isEnabled()) {
            val message = new ResponseMessage(config, requestMethod, requestPath, response);
            log.info(toFormattable(message));
        }
    }

    @Override
    public void logResponse(HttpMethod requestMethod, String requestPath, Throwable e) {
        if (config.isEnabled()) {
            log.error("Request processing failed: {} {}", requestMethod, requestPath, e);
        }
    }
}
