package ru.yandex.mail.micronaut.http_logger.client;

import io.micronaut.http.HttpAttributes;
import io.micronaut.http.HttpResponse;
import io.micronaut.http.MutableHttpRequest;
import io.micronaut.http.annotation.Filter;
import io.micronaut.http.filter.ClientFilterChain;
import io.micronaut.http.filter.HttpClientFilter;
import lombok.AllArgsConstructor;
import lombok.val;
import org.reactivestreams.Publisher;
import reactor.core.publisher.Mono;

import javax.inject.Inject;

@Filter("/**")
@AllArgsConstructor(onConstructor_=@Inject)
public class HttpClientResponseLoggingFilter implements HttpClientFilter {
    private final ClientLoggerManager manager;

    private Publisher<? extends HttpResponse<?>> filter(String clientId, MutableHttpRequest<?> request,
                                                        ClientFilterChain chain) {
        val logger = manager.findLogger(clientId);

        if (logger == null) {
            return chain.proceed(request);
        }

        val method = request.getMethod();
        val path = request.getPath();

        return Mono.from(chain.proceed(request))
            .doOnError(e -> logger.logResponse(method, path, e))
            .doOnSuccess(response -> logger.logResponse(method, path, response));
    }

    @Override
    public Publisher<? extends HttpResponse<?>> doFilter(MutableHttpRequest<?> request, ClientFilterChain chain) {
        return request.getAttribute(HttpAttributes.SERVICE_ID, String.class)
            .<Publisher<? extends HttpResponse<?>>>map(clientId -> filter(clientId, request, chain))
            .orElseGet(() -> chain.proceed(request));
    }

    @Override
    public int getOrder() {
        return HIGHEST_PRECEDENCE;
    }
}
