package ru.yandex.mail.micronaut.http_logger.message;

import lombok.AccessLevel;
import lombok.experimental.FieldDefaults;
import lombok.val;
import one.util.streamex.StreamEx;
import ru.yandex.mail.micronaut.http_logger.config.LoggingOptions;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static java.util.Objects.requireNonNullElseGet;
import static ru.yandex.mail.micronaut.http_logger.Constants.BODY_DELIMITER;
import static ru.yandex.mail.micronaut.http_logger.Constants.CUT_LIST;
import static ru.yandex.mail.micronaut.http_logger.Constants.DEFAULT_SENSITIVE_HEADERS;

@FieldDefaults(makeFinal = true, level = AccessLevel.PRIVATE)
public class MessageBuilder {
    Set<String> sensitiveHeaders;
    Set<String> sensitiveParams;

    private boolean isSensitiveHeader(String name) {
        return sensitiveHeaders.contains(name);
    }

    private boolean isSensitiveParam(String name) {
        return sensitiveParams.contains(name);
    }

    private static void appendLine(StringBuilder builder, String line) {
        builder.append(line).append('\n');
    }

    private static void appendLine(StringBuilder builder, Runnable appender) {
        appender.run();
        builder.append('\n');
    }

    private static <T> void appendList(StringBuilder builder, List<T> list, String delimiter) {
        if (list.isEmpty()) {
            return;
        }

        for (T value : list) {
            builder.append(value)
                .append(delimiter);
        }

        builder.setLength(builder.length() - delimiter.length());
    }

    private static void appendBody(StringBuilder builder, String body) {
        appendLine(builder, BODY_DELIMITER);
        appendLine(builder, body);
        appendLine(builder, BODY_DELIMITER);
    }

    private void appendHeaders(StringBuilder builder, Map<String, List<String>> headers) {
        headers.forEach((name, values) -> {
            val safeValues = isSensitiveHeader(name) ? CUT_LIST : values;
            appendLine(builder, () -> {
                builder.append(name).append(": ");
                appendList(builder, safeValues, ", ");
            });
        });
    }

    private void appendParameters(StringBuilder builder, Map<String, List<String>> parameters) {
        if (parameters.isEmpty()) {
            return;
        }

        builder.append('?');

        parameters.forEach((name, values) -> {
            val safeValues = isSensitiveParam(name) ? CUT_LIST : values;
            builder.append(name).append('=');
            appendList(builder, safeValues, ",");
            builder.append('&');
        });

        builder.setLength(builder.length() - "&".length());
    }

    public MessageBuilder(LoggingOptions config) {
        this.sensitiveHeaders = StreamEx.of(DEFAULT_SENSITIVE_HEADERS)
            .append(requireNonNullElseGet(config.getSensitiveHeaders(), Collections::emptySet))
            .toImmutableSet();
        this.sensitiveParams = requireNonNullElseGet(config.getSensitiveParams(), Collections::emptySet);
    }

    public void buildMessage(StringBuilder builder, RequestMessage message) {
        appendLine(builder, () -> {
            builder.append(message.getAction())
                .append(" HTTP request: ")
                .append(message.getMethod())
                .append(' ')
                .append(message.getPath());
            appendParameters(builder, message.getParams());
        });

        appendHeaders(builder, message.getHeaders());
        appendBody(builder, message.getBody());
    }

    public void buildMessage(StringBuilder builder, ResponseMessage message) {
        appendLine(builder, () -> {
            builder.append("Response for: ")
                .append(message.getRequestMethod())
                .append(' ')
                .append(message.getRequestPath());
        });

        appendLine(builder, () -> {
            val status = message.getStatus();
            builder.append("Status code: ").append(status.getCode()).append(" (").append(status).append(")");
        });

        appendHeaders(builder, message.getHeaders());
        appendBody(builder, message.getBody());
    }
}
