package ru.yandex.mail.micronaut.http_logger.slf4j;

import io.micronaut.http.HttpMethod;
import io.micronaut.http.HttpRequest;
import io.micronaut.http.HttpResponse;
import lombok.AccessLevel;
import lombok.experimental.FieldDefaults;
import lombok.val;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import ru.yandex.mail.micronaut.http_logger.RequestLogger;
import ru.yandex.mail.micronaut.http_logger.config.LoggingOptions;
import ru.yandex.mail.micronaut.http_logger.message.MessageBuilder;
import ru.yandex.mail.micronaut.http_logger.message.RequestMessage;
import ru.yandex.mail.micronaut.http_logger.message.ResponseMessage;

@FieldDefaults(makeFinal = true, level = AccessLevel.PRIVATE)
final class Slf4JRequestLogger implements RequestLogger {
    private static final int INITIAL_BUILDER_CAPACITY = 512;

    @SuppressWarnings("NonConstantLogger")
    Logger log;
    LoggingOptions config;
    MessageBuilder messageBuilder;

    Slf4JRequestLogger(String loggerName, LoggingOptions config) {
        this.config = config;
        this.log = LoggerFactory.getLogger(loggerName);
        this.messageBuilder = new MessageBuilder(config);
    }

    @Override
    public <B> void logRequest(HttpRequest<B> request, String action) {
        if (!config.isEnabled() || !log.isInfoEnabled()) {
            return;
        }

        val message = new RequestMessage(config, request, action);
        val builder = new StringBuilder(INITIAL_BUILDER_CAPACITY);
        messageBuilder.buildMessage(builder, message);

        log.info("{}", builder);
    }

    @Override
    public <B> void logResponse(HttpMethod requestMethod, String requestPath, HttpResponse<B> response) {
        if (!config.isEnabled() || !log.isInfoEnabled()) {
            return;
        }

        val message = new ResponseMessage(config, requestMethod, requestPath, response);
        val builder = new StringBuilder(INITIAL_BUILDER_CAPACITY);
        messageBuilder.buildMessage(builder, message);

        log.info("{}", builder);
    }

    @Override
    public void logResponse(HttpMethod requestMethod, String requestPath, Throwable e) {
        if (config.isEnabled()) {
            log.error("Request processing failed: {} {}", requestMethod, requestPath, e);
        }
    }
}
