package ru.yandex.mail.micronaut.micrometer;

import io.micrometer.core.instrument.Meter;
import io.micrometer.core.instrument.Tag;
import io.micrometer.core.instrument.Tags;
import io.micrometer.core.instrument.config.MeterFilter;
import io.micronaut.configuration.metrics.annotation.RequiresMetrics;
import io.micronaut.context.annotation.Bean;
import io.micronaut.context.annotation.Factory;
import lombok.val;
import one.util.streamex.StreamEx;

import javax.annotation.Nonnull;
import javax.inject.Singleton;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Predicate;

import static io.micronaut.configuration.metrics.binder.web.WebMetricsPublisher.METRIC_HTTP_CLIENT_REQUESTS;
import static io.micronaut.configuration.metrics.binder.web.WebMetricsPublisher.METRIC_HTTP_SERVER_REQUESTS;
import static java.util.function.Predicate.not;

@Factory
@RequiresMetrics
public class MeterFilterFactory {
    private static final Set<String> MICRONAUT_HTTP_METER_NAMES = Set.of(
        METRIC_HTTP_SERVER_REQUESTS,
        METRIC_HTTP_CLIENT_REQUESTS
    );

    private static MeterFilter replaceTagValues(Set<String> meterNames, String tagKey, Function<String, String> replacement) {
        return new MeterFilter() {
            private final MeterFilter filter = MeterFilter.replaceTagValues(tagKey, replacement);

            @Override
            @Nonnull
            public Meter.Id map(@Nonnull Meter.Id id) {
                if (meterNames.contains(id.getName())) {
                    return filter.map(id);
                } else {
                    return id;
                }
            }
        };
    }

    private static MeterFilter filterTags(Set<String> meterNames, Predicate<Tag> includePredicate) {
        return new MeterFilter() {
            @Override
            @Nonnull
            public Meter.Id map(@Nonnull Meter.Id id) {
                if (meterNames.contains(id.getName())) {
                    val tags = StreamEx.of(id.getTagsAsIterable().spliterator())
                        .filter(includePredicate)
                        .toArray(Tag[]::new);
                    return id.replaceTags(Tags.of(tags));
                } else {
                    return id;
                }
            }
        };
    }

    private static MeterFilter excludeTags(Set<String> meterNames, Set<String> tagKeys) {
        return filterTags(meterNames, not(tag -> tagKeys.contains(tag.getKey())));
    }

    private static MeterFilter includeTags(Set<String> meterNames, Set<String> tagKeys) {
        return filterTags(meterNames, tag -> tagKeys.contains(tag.getKey()));
    }

    @Bean
    @Singleton
    public MeterFilter ignoreUnistatEndpoint() {
        return MeterFilter.deny(id -> {
            return id.getName().equals(METRIC_HTTP_SERVER_REQUESTS) &&
                "/unistat".equals(id.getTag("uri"));
        });
    }

    @Bean
    @Singleton
    public MeterFilter micronautStatusCodesReducer() {
        val httpStatusCodeLength = 3;
        val httpCodeMapping = new String[] { "", "1xx", "2xx", "3xx", "4xx", "5xx" };

        return replaceTagValues(MICRONAUT_HTTP_METER_NAMES, "status", value -> {
            if (value.length() == httpStatusCodeLength) {
                val index = Character.digit(value.charAt(0), 10);
                return (index > 0 && index < httpCodeMapping.length) ? httpCodeMapping[index] : value;
            } else {
                return value;
            }
        });
    }

    @Bean
    @Singleton
    public MeterFilter micronautIgnoreMethodTagOn404() {
        return new MeterFilter() {
            private boolean isNotFound(Meter.Id id) {
                val uriTagValue = id.getTag("uri");
                return "NOT_FOUND".equals(uriTagValue);
            }

            @Nonnull
            @Override
            public Meter.Id map(@Nonnull Meter.Id id) {
                if (id.getName().equals(METRIC_HTTP_SERVER_REQUESTS) && isNotFound(id)) {
                    val tags = StreamEx.of(id.getTagsAsIterable().spliterator())
                        .filter(not(tag -> tag.getKey().equals("method")))
                        .toArray(Tag[]::new);
                    return id.replaceTags(Tags.of(tags));
                } else {
                    return id;
                }
            }
        };
    }

    @Bean
    @Singleton
    public MeterFilter micronautHttpTagsFilter() {
        return includeTags(MICRONAUT_HTTP_METER_NAMES, Set.of("method", "status", "uri"));
    }

    @Bean
    @Singleton
    public MeterFilter micronautRedundantSignalsFilter() {
        val signalNames = Set.of(
            "jvm.threads.peak",
            "jvm.threads.daemon",
            "jvm.threads.live"
        );
        return MeterFilter.deny(id -> signalNames.contains(id.getName()));
    }

    @Bean
    @Singleton
    public MeterFilter micronautGcTagsFilter() {
        val signalNames = Set.of(
            "jvm.gc.concurrent.phase.time",
            "jvm.gc.pause"
        );
        val tags = Set.of("action", "cause");
        return excludeTags(signalNames, tags);
    }
}
