package ru.yandex.mail.micronaut.micrometer.unistat;

import io.micrometer.core.instrument.Clock;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.DistributionSummary;
import io.micrometer.core.instrument.FunctionCounter;
import io.micrometer.core.instrument.FunctionTimer;
import io.micrometer.core.instrument.Gauge;
import io.micrometer.core.instrument.LongTaskTimer;
import io.micrometer.core.instrument.Measurement;
import io.micrometer.core.instrument.Meter;
import io.micrometer.core.instrument.MeterRegistry;
import io.micrometer.core.instrument.Timer;
import io.micrometer.core.instrument.cumulative.CumulativeCounter;
import io.micrometer.core.instrument.cumulative.CumulativeDistributionSummary;
import io.micrometer.core.instrument.cumulative.CumulativeFunctionCounter;
import io.micrometer.core.instrument.cumulative.CumulativeFunctionTimer;
import io.micrometer.core.instrument.cumulative.CumulativeTimer;
import io.micrometer.core.instrument.distribution.DistributionStatisticConfig;
import io.micrometer.core.instrument.distribution.pause.PauseDetector;
import io.micrometer.core.instrument.internal.DefaultGauge;
import io.micrometer.core.instrument.internal.DefaultLongTaskTimer;
import io.micrometer.core.instrument.internal.DefaultMeter;
import io.micronaut.context.annotation.Requires;
import lombok.val;
import one.util.streamex.StreamEx;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.inject.Inject;
import javax.inject.Singleton;
import java.time.Duration;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.function.ToDoubleFunction;
import java.util.function.ToLongFunction;

import static io.micronaut.configuration.metrics.micrometer.MeterRegistryFactory.MICRONAUT_METRICS_ENABLED;

@Singleton
@Requires(property = MICRONAUT_METRICS_ENABLED, notEquals = "false")
public class UnistatMeterRegistry extends MeterRegistry {
    public static final TimeUnit BASE_TIME_UNIT = TimeUnit.MILLISECONDS;

    private final UnistatMeterRegistryConfiguration configuration;
    private final List<UnistatSlaConfiguration> slaConfigs;

    private DistributionStatisticConfig resolveConfig(Meter.Id id, DistributionStatisticConfig defaultConfig) {
        return StreamEx.of(slaConfigs)
            .findFirst(s -> s.matches(id))
            .map(UnistatSlaConfiguration::getValues)
            .map(sla -> {
                return DistributionStatisticConfig.builder()
                    .sla(sla.mapToLong(Duration::toNanos).toArray())
                    .build()
                    .merge(defaultConfig);
            })
            .orElse(defaultConfig);
    }

    @Inject
    public UnistatMeterRegistry(UnistatMeterRegistryConfiguration configuration, List<UnistatSlaConfiguration> slaConfigs) {
        super(Clock.SYSTEM);
        this.configuration = configuration;
        this.slaConfigs = slaConfigs;
    }

    @Override
    @Nonnull
    protected <T> Gauge newGauge(@Nonnull Meter.Id id, @Nullable T obj, @Nonnull ToDoubleFunction<T> valueFunction) {
        return new DefaultGauge<>(id, obj, valueFunction);
    }

    @Override
    @Nonnull
    protected Counter newCounter(@Nonnull Meter.Id id) {
        return new CumulativeCounter(id);
    }

    @Override
    @Nonnull
    protected LongTaskTimer newLongTaskTimer(@Nonnull Meter.Id id) {
        return new DefaultLongTaskTimer(id, Clock.SYSTEM);
    }

    @Override
    @Nonnull
    protected Timer newTimer(@Nonnull Meter.Id id, @Nonnull DistributionStatisticConfig distributionStatisticConfig,
                             @Nonnull PauseDetector pauseDetector) {
        val config = resolveConfig(id, distributionStatisticConfig);
        return new CumulativeTimer(id, Clock.SYSTEM, config, pauseDetector, getBaseTimeUnit());
    }

    @Override
    @Nonnull
    protected DistributionSummary newDistributionSummary(@Nonnull Meter.Id id,
                                                         @Nonnull DistributionStatisticConfig distributionStatisticConfig,
                                                         double scale) {
        val config = resolveConfig(id, distributionStatisticConfig);
        return new CumulativeDistributionSummary(id, Clock.SYSTEM, config, scale, false);
    }

    @Override
    @Nonnull
    protected Meter newMeter(@Nonnull Meter.Id id, @Nonnull Meter.Type type, @Nonnull Iterable<Measurement> measurements) {
        return new DefaultMeter(id, type, measurements);
    }

    @Override
    @Nonnull
    protected <T> FunctionTimer newFunctionTimer(@Nonnull Meter.Id id, @Nonnull T obj, @Nonnull ToLongFunction<T> countFunction,
                                                 @Nonnull ToDoubleFunction<T> totalTimeFunction, @Nonnull TimeUnit totalTimeFunctionUnit) {
        return new CumulativeFunctionTimer<>(id, obj, countFunction, totalTimeFunction, totalTimeFunctionUnit, getBaseTimeUnit());
    }

    @Override
    @Nonnull
    protected <T> FunctionCounter newFunctionCounter(@Nonnull Meter.Id id, @Nonnull T obj,
                                                     @Nonnull ToDoubleFunction<T> countFunction) {
        return new CumulativeFunctionCounter<>(id, obj, countFunction);
    }

    @Override
    @Nonnull
    protected TimeUnit getBaseTimeUnit() {
        return BASE_TIME_UNIT;
    }

    @Override
    @Nonnull
    protected DistributionStatisticConfig defaultHistogramConfig() {
        return DistributionStatisticConfig.builder()
            .percentilesHistogram(true)
            .sla(configuration.getDefaultSlaNanos())
            .expiry(configuration.getExpiry())
            .bufferLength(configuration.getBufferLength())
            .build();
    }
}
