package ru.yandex.mail.micronaut.micrometer.unistat;

import com.fasterxml.jackson.core.JsonEncoding;
import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonGenerator;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.DistributionSummary;
import io.micrometer.core.instrument.FunctionCounter;
import io.micrometer.core.instrument.FunctionTimer;
import io.micrometer.core.instrument.Gauge;
import io.micrometer.core.instrument.LongTaskTimer;
import io.micrometer.core.instrument.Meter;
import io.micrometer.core.instrument.MeterRegistry;
import io.micrometer.core.instrument.Tag;
import io.micrometer.core.instrument.Timer;
import io.micrometer.core.instrument.distribution.HistogramSupport;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import one.util.streamex.StreamEx;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.IdentityHashMap;
import java.util.Map;
import java.util.regex.Pattern;

import static ru.yandex.mail.micronaut.micrometer.unistat.UnistatMeterRegistry.BASE_TIME_UNIT;

@Slf4j
class UnistatSerializer {
    @FunctionalInterface
    private interface IORunnable {
        void run() throws IOException;
    }

    private static final int INITIAL_CAPACITY = 200;
    private static final Pattern SIGNAL_NAME_INVALID_CHARACTERS_PATTERN = Pattern.compile("[^a-zA-Z0-9_]");
    private static final Pattern SMOOTH_NAME_PATTERN = Pattern.compile("__+");

    private final ByteArrayOutputStream outStream;
    private final JsonGenerator generator;
    private final Map<Meter, String> meterNameCache = new IdentityHashMap<>();

    @SneakyThrows
    UnistatSerializer(JsonFactory factory) {
        this.outStream = new ByteArrayOutputStream(INITIAL_CAPACITY);
        this.generator = factory.createGenerator(outStream, JsonEncoding.UTF8);
    }

    private static String resolveName(Meter.Id id) {
        val tags = StreamEx.of(id.getTagsAsIterable().spliterator())
            .map(Tag::getValue)
            .joining("_");

        val dirtyName = String.join("_", id.getName(), tags);
        val normalizedName = SIGNAL_NAME_INVALID_CHARACTERS_PATTERN.matcher(dirtyName).replaceAll("_");
        return SMOOTH_NAME_PATTERN.matcher(normalizedName)
            .replaceAll("_")
            .replaceAll("^_+", "")
            .replaceAll("_+$", "")
            .toLowerCase();
    }

    private String getName(Meter meter) {
        return meterNameCache.computeIfAbsent(meter, m -> resolveName(m.getId()));
    }

    @SneakyThrows
    private void surroundWithArray(IORunnable runnable) {
        generator.writeStartArray();
        runnable.run();
        generator.writeEndArray();
    }

    private void writeHistogramBucket(double bucketStart, long count) {
        surroundWithArray(() -> {
            generator.writeNumber(bucketStart);
            generator.writeNumber(count);
        });
    }

    private void writeHistogram(String name, HistogramSupport histogram) {
        val snapshot = histogram.takeSnapshot();

        surroundWithArray(() -> {
            generator.writeString(name + "_count" + Suffix.SUMM);
            generator.writeNumber(snapshot.count());
        });

        surroundWithArray(() -> {
            generator.writeString(name + "_time" + Suffix.HISTOGRAM);
            surroundWithArray(() -> {
                var bucketStart = 0.0;
                var previousBucketValue = 0L;
                for (val countAtBucket : snapshot.histogramCounts()) {
                    val bucketSize = (long) countAtBucket.count();
                    writeHistogramBucket(bucketStart, bucketSize - previousBucketValue);
                    bucketStart = countAtBucket.bucket(BASE_TIME_UNIT);
                    previousBucketValue = bucketSize;
                }
                writeHistogramBucket(bucketStart, 0L);
            });
        });
    }

    private void writeGauge(Gauge gauge) {
        surroundWithArray(() -> {
            generator.writeString(getName(gauge) + Suffix.ABS_MAX_MAX_MAX);
            generator.writeNumber(gauge.value());
        });
    }

    private void writeCounter(Counter counter) {
        surroundWithArray(() -> {
            generator.writeString(getName(counter) + Suffix.SUMM);
            generator.writeNumber(counter.count());
        });
    }

    private void writeLongTaskTimer(LongTaskTimer timer) {
        val name = getName(timer);

        surroundWithArray(() -> {
            generator.writeString(name + "_active_count" + Suffix.ABS_SUM_SUM_SUM);
            generator.writeNumber(timer.activeTasks());
        });

        surroundWithArray(() -> {
            generator.writeString(name + "_active_total_duration" + Suffix.ABS_SUM_SUM_SUM);
            generator.writeNumber(timer.duration(BASE_TIME_UNIT));
        });
    }

    private void writeTimer(Timer timer) {
        writeHistogram(getName(timer), timer);
    }

    private void writeDistributionSummary(DistributionSummary summary) {
        writeHistogram(getName(summary), summary);
    }

    private void writeMeter(Meter meter) {
        log.error("Unknown {} meter named {} found", meter.getClass().getSimpleName(), getName(meter));
    }

    private void writeFunctionTimer(FunctionTimer timer) {
        val name = getName(timer);

        surroundWithArray(() -> {
            generator.writeString(name + "_count" + Suffix.SUMM);
            generator.writeNumber(timer.count());
        });

        surroundWithArray(() -> {
            generator.writeString(name + "_total_time" + Suffix.HISTOGRAM);
            generator.writeNumber(timer.totalTime(timer.baseTimeUnit()));
        });
    }

    private void writeFunctionCounter(FunctionCounter counter) {
        surroundWithArray(() -> {
            generator.writeString(getName(counter) + Suffix.SUMM);
            generator.writeNumber(counter.count());
        });
    }

    @SneakyThrows
    private void writePrologue() {
        generator.writeStartArray();
    }

    @SneakyThrows
    private void writeEpilogue() {
        generator.writeEndArray();
    }

    @SneakyThrows
    String serialize(MeterRegistry registry) {
        outStream.reset();
        writePrologue();
        registry.forEachMeter(meter -> meter.use(
            this::writeGauge,
            this::writeCounter,
            this::writeTimer,
            this::writeDistributionSummary,
            this::writeLongTaskTimer,
            this::writeGauge,
            this::writeFunctionCounter,
            this::writeFunctionTimer,
            this::writeMeter
        ));
        writeEpilogue();
        generator.flush();
        return outStream.toString(StandardCharsets.UTF_8);
    }
}
