package ru.yandex.mail.micronaut.micrometer.unistat;

import io.micrometer.core.instrument.Meter;
import io.micronaut.configuration.metrics.annotation.RequiresMetrics;
import io.micronaut.context.annotation.ConfigurationProperties;
import io.micronaut.context.annotation.Context;
import io.micronaut.context.annotation.EachProperty;
import io.micronaut.core.annotation.Introspected;
import lombok.Data;
import lombok.val;
import one.util.streamex.IntStreamEx;
import one.util.streamex.StreamEx;

import javax.annotation.Nullable;
import javax.annotation.PostConstruct;
import javax.validation.ValidationException;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import javax.validation.constraints.Positive;
import java.time.Duration;
import java.util.List;
import java.util.Optional;
import java.util.regex.Pattern;

@Data
@Context
@Introspected
@RequiresMetrics
@EachProperty("unistat.sla")
public class UnistatSlaConfiguration {
    @Data
    @Introspected
    @ConfigurationProperties("generator")
    public static class Generator {
        @NotNull @Positive Duration left;
        @NotNull @Positive Duration pivot;
        @NotNull @Positive Duration right;
        @NotNull @Positive Duration timeout;

        private void validate() {
            if (left != null && pivot != null && left.compareTo(pivot) >= 0) {
                throw new ValidationException("'left' needs to be less than 'pivot'");
            }

            if (pivot != null && right != null && pivot.compareTo(right) >= 0) {
                throw new ValidationException("'pivot' needs to be less than 'right'");
            }
        }

        private static StreamEx<Duration> genRange(Duration left, Duration right, int count) {
            val step = right.minus(left).dividedBy(count);
            return IntStreamEx.range(count)
                .mapToObj(i -> step.multipliedBy(i).plus(left));
        }

        public StreamEx<Duration> generate() {
            val timeoutNanos = timeout.toNanos();
            return genRange(Duration.ZERO, left, 5)
                .remove(Duration::isZero) // micrometer requires positive only values
                .append(genRange(left, pivot, 20))
                .append(genRange(pivot, right, 15))
                .append(genRange(right, timeout, 5))
                .append(IntStreamEx.range(5).mapToObj(i -> {
                    val nanos = timeoutNanos * (0.2 * i + 1);
                    return Duration.ofNanos(Math.round(nanos));
                }))
                .distinct();
        }
    }

    Optional<String> metricName = Optional.empty();
    Optional<String> tagKey = Optional.empty();
    Optional<String> tagValue = Optional.empty();
    Optional<@NotEmpty List<Duration>> fixed = Optional.empty();
    Optional<Generator> generator = Optional.empty();

    public void setGenerator(@Nullable Generator generator) {
        this.generator = Optional.ofNullable(generator);
    }

    @PostConstruct
    private void validate() {
        if (metricName.isEmpty() && tagKey.isEmpty()) {
            throw new ValidationException("'metric-name' and 'tag-key' both is empty");
        }

        if (tagValue.isPresent() && tagKey.isEmpty()) {
            throw new ValidationException("'tag-value' requires 'tag-key' to be set");
        }

        if (fixed.isEmpty() == generator.isEmpty()) {
            throw new ValidationException("Expected 'fixed' or 'generator' to be set, but not both");
        }

        fixed.ifPresent(value -> {
            if (value.isEmpty()) {
                throw new ValidationException("'fixed' needs to be non empty");
            }
        });

        generator.ifPresent(Generator::validate);
    }

    private static boolean matches(String pattern, String value) {
        return Pattern.compile(pattern).matcher(value).find();
    }

    public boolean matches(Meter.Id id) {
        if (metricName.isPresent()) {
            if (!matches(metricName.get(), id.getName())) {
                return false;
            }
        }

        if (tagKey.isPresent()) {
            return StreamEx.of(id.getTagsAsIterable().spliterator())
                .anyMatch(tag -> {
                    return matches(tagKey.get(), tag.getKey()) &&
                        (tagValue.isEmpty() || matches(tagValue.get(), tag.getValue()));
                });
        }

        return true;
    }

    public StreamEx<Duration> getValues() {
        return fixed.map(StreamEx::of)
            .or(() -> generator.map(Generator::generate))
            .orElseThrow();
    }
}
