package ru.yandex.mail.micronaut.tvm;

import io.micronaut.context.annotation.Requires;
import io.micronaut.http.HttpAttributes;
import io.micronaut.http.HttpResponse;
import io.micronaut.http.MutableHttpRequest;
import io.micronaut.http.filter.ClientFilterChain;
import io.micronaut.http.filter.HttpClientFilter;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.reactivestreams.Publisher;
import ru.yandex.mail.micronaut.tvm.auth.TvmSecured;

import javax.inject.Inject;
import javax.inject.Singleton;

import static ru.yandex.mail.micronaut.tvm.Constants.TVM_SERVICE_TICKET_HEADER;

@Slf4j
@Singleton
@TvmSecured
@Requires(beans = TvmManager.class)
@AllArgsConstructor(onConstructor_=@Inject)
public class ClientTvmFilter implements HttpClientFilter {
    private final TvmManager tvmManager;

    private static MutableHttpRequest<?> addTvmHeader(MutableHttpRequest<?> request, String ticket) {
        return request.header(TVM_SERVICE_TICKET_HEADER, ticket);
    }

    @Override
    public Publisher<? extends HttpResponse<?>> doFilter(MutableHttpRequest<?> request, ClientFilterChain chain) {
        val clientId = request.getAttribute(HttpAttributes.SERVICE_ID, String.class).orElse("");
        if (clientId.isEmpty()) {
            log.warn("TVM processing is impossible: HTTP client doesn't have a service id");
            return chain.proceed(request);
        }

        return tvmManager.getTicketFor(clientId)
            .map(ticket -> addTvmHeader(request, ticket.getPayload()))
            .flatMapMany(chain::proceed);
    }
}
