package ru.yandex.mail.micronaut.tvm;

import io.micronaut.context.annotation.Requires;
import lombok.val;
import one.util.streamex.EntryStream;
import reactor.core.publisher.Mono;
import ru.yandex.mail.micronaut.tvm.client.TvmClient;
import ru.yandex.mail.micronaut.tvm.client.TvmTicket.ServiceTvmTicket;
import ru.yandex.mail.micronaut.tvm.exceptions.TicketNotFoundException;

import javax.inject.Inject;
import javax.inject.Singleton;
import java.util.Map;
import java.util.Optional;

@Singleton
@Requires(beans = TvmConfiguration.class)
public class DefaultTvmManager implements TvmManager {
    private final Map<String, Long> services;
    private final Map<Long, String> clientNames;
    private final TvmClient tvmClient;

    private static Map<Long, String> getClientNames(TvmConfiguration configuration) {
        return EntryStream.of(configuration.getAllowedClients())
            .invert()
            .distinctKeys()
            .toImmutableMap();
    }

    @Inject
    public DefaultTvmManager(TvmConfiguration configuration, TvmClient tvmClient) {
        this.services = configuration.getServices();
        this.clientNames = getClientNames(configuration);
        this.tvmClient = tvmClient;
    }

    @Override
    public Mono<ServiceTvmTicket> getTicketFor(String clientId) {
        val id = services.get(clientId);
        if (id == null) {
            return Mono.error(new TicketNotFoundException(clientId));
        }

        return tvmClient.getTicketFor(id);
    }

    @Override
    public Optional<String> getClientName(long tvmId) {
        return Optional.ofNullable(clientNames.get(tvmId));
    }
}
