package ru.yandex.mail.micronaut.tvm.auth;

import io.micronaut.context.annotation.Requires;
import io.micronaut.http.HttpRequest;
import io.micronaut.security.authentication.Authentication;
import io.micronaut.security.authentication.AuthenticationException;
import io.micronaut.security.filters.AuthenticationFetcher;
import lombok.val;
import org.reactivestreams.Publisher;
import reactor.core.publisher.Mono;
import ru.yandex.mail.micronaut.tvm.TvmManager;
import ru.yandex.mail.micronaut.tvm.client.CheckResult;
import ru.yandex.mail.micronaut.tvm.client.TvmClient;
import ru.yandex.mail.micronaut.tvm.client.TvmTicket;

import javax.inject.Inject;
import javax.inject.Singleton;

import java.util.Optional;

import static ru.yandex.mail.micronaut.tvm.Constants.TVM_SERVICE_TICKET_HEADER;
import static ru.yandex.mail.micronaut.tvm.Constants.TVM_USER_TICKET_HEADER;

@Singleton
@Requires(beans = TvmClient.class)
public class TvmAuthenticationFetcher implements AuthenticationFetcher {
    @Inject
    private TvmClient tvmClient;

    @Inject
    private TvmManager tvmManager;

    private String getServiceName(TvmTicket.ServiceTvmTicket ticket) {
        return tvmManager.getClientName(ticket.getTvmId())
            .orElseThrow(() -> new AuthenticationException("Client name not found but authorization has been succeeded"));
    }

    private static <T extends TvmTicket> Mono<T> processResult(Mono<CheckResult<T>> checkResult) {
        return checkResult
            .filter(CheckResult::isSuccess)
            .map(CheckResult::getTicket);
    }

    private Authentication buildAuthentication(TvmTicket.ServiceTvmTicket serviceTicket,
                                               Optional<TvmTicket.UserTvmTicket> userTicket) {
        return new TvmAuthentication(getServiceName(serviceTicket), serviceTicket, userTicket);
    }

    @Override
    public Publisher<Authentication> fetchAuthentication(HttpRequest<?> request) {
        val headers = request.getHeaders();
        val serviceTicketPayload = headers.get(TVM_SERVICE_TICKET_HEADER);
        val userTicketPayload = headers.get(TVM_USER_TICKET_HEADER);

        if (serviceTicketPayload == null) {
            return Mono.empty();
        }

        val serviceTicketMono = processResult(tvmClient.checkServiceTicket(serviceTicketPayload));
        val userTicketMono = userTicketPayload == null
            ? Mono.just(Optional.<TvmTicket.UserTvmTicket>empty())
            : processResult(tvmClient.checkUserTicket(userTicketPayload)).map(Optional::of);

        return Mono.zip(serviceTicketMono, userTicketMono, this::buildAuthentication);
    }
}
