package ru.yandex.mail.micronaut.tvm.tvmtool;

import io.micronaut.context.annotation.Requires;
import io.micronaut.http.HttpStatus;
import io.micronaut.http.client.exceptions.HttpClientResponseException;
import io.micronaut.http.exceptions.HttpException;
import lombok.val;
import one.util.streamex.EntryStream;
import one.util.streamex.LongStreamEx;
import one.util.streamex.StreamEx;
import reactor.core.publisher.Mono;
import ru.yandex.mail.micronaut.tvm.client.CheckResult;
import ru.yandex.mail.micronaut.tvm.client.TvmClient;
import ru.yandex.mail.micronaut.tvm.client.TvmTicket;
import ru.yandex.mail.micronaut.tvm.client.TvmTicket.ServiceTvmTicket;
import ru.yandex.mail.micronaut.tvm.client.TvmTicket.UserTvmTicket;
import ru.yandex.mail.micronaut.tvm.exceptions.TicketNotFoundException;
import ru.yandex.mail.micronaut.tvm.exceptions.TvmToolException;

import javax.inject.Inject;
import javax.inject.Singleton;
import java.util.Map;
import java.util.function.Function;

import static java.util.Objects.requireNonNullElseGet;
import static java.util.function.Predicate.not;

@Singleton
@Requires(beans = HttpTvmToolClient.class)
public class TvmToolClient implements TvmClient {
    @Inject
    private HttpTvmToolClient client;

    private static ServiceTvmTicket parseTicket(TicketResponse response) {
        return new ServiceTvmTicket(response.getTicket(), response.getTvmId());
    }

    private static Map<Long, ServiceTvmTicket> parseTickets(Map<String, TicketResponse> response) {
        return EntryStream.of(response)
            .mapValues(TvmToolClient::parseTicket)
            .mapToKey((key, value) -> value.getTvmId())
            .toImmutableMap();
    }

    private static Throwable wrapHttpException(Throwable e) {
        if (e instanceof HttpClientResponseException) {
            val ex = (HttpClientResponseException) e;
            return ex.getResponse()
                .getBody(String.class)
                .map(msg -> new TvmToolException(msg, ex))
                .orElseGet(() -> new TvmToolException(ex));
        } else if (e instanceof HttpException) {
            return new TvmToolException(e);
        } else {
            return e;
        }
    }

    private static boolean isForbidden(Throwable e) {
        return e instanceof HttpClientResponseException
            && ((HttpClientResponseException) e).getStatus() == HttpStatus.FORBIDDEN;
    }

    private static Function<? super Throwable, ? extends Throwable> fixCompiler(Function<Throwable, Throwable> func) {
        return func;
    }

    @SuppressWarnings("unchecked")
    private static <T extends TvmTicket, R> Mono<CheckResult<T>> checkTicket(Mono<R> response,
                                                                             Function<R, T> ticketExtractor) {
        return response
            .map(ticketExtractor)
            .<CheckResult<T>>map(CheckResult.Success::new)
            .onErrorResume(TvmToolClient::isForbidden, ignore -> Mono.<CheckResult<T>>just(CheckResult.Forbidden))
            .onErrorMap(fixCompiler(TvmToolClient::wrapHttpException));
    }

    @Override
    public Mono<ServiceTvmTicket> getTicketFor(long dst) {
        return getTicketsFor(dst)
            .map(tickets -> {
                return requireNonNullElseGet(tickets.get(dst), () -> { throw new TicketNotFoundException(dst); });
            });
    }

    @Override
    public Mono<Map<Long, ServiceTvmTicket>> getTicketsFor(long... dst) {
        val dstList = LongStreamEx.of(dst).boxed().toImmutableList();
        return client.tickets(dstList)
            .map(TvmToolClient::parseTickets)
            .onErrorMap(TvmToolClient::wrapHttpException)
            .doOnSuccess(tickets -> {
                if (tickets.size() != dst.length) {
                    val missingDsts = StreamEx.of(dstList)
                        .filter(not(tickets::containsKey))
                        .joining(",");
                    throw new TvmToolException("Bad response: " + missingDsts + " is missing");
                }
            });
    }

    @Override
    public Mono<CheckResult<ServiceTvmTicket>> checkServiceTicket(String ticket) {
        return checkTicket(
            client.checkService(ticket),
            response -> new ServiceTvmTicket(ticket, response.getSrc())
        );
    }

    @Override
    public Mono<CheckResult<UserTvmTicket>> checkUserTicket(String ticket) {
        return checkTicket(
            client.checkUser(ticket),
            response -> new UserTvmTicket(ticket, response.getDefaultUid())
        );
    }
}
