#ifndef EDP_CHUNKS_H_
#define EDP_CHUNKS_H_

#include <mimeparser/Chunk.h>
#include <mimeparser/chunks_iterator.h>
#include <vector>
#include <iterator>

namespace MimeParser {

/**
 * This is a chunk storage - a collectios of chunks,
 * separate iterator ranges, that can be used as one
 * continuous range.
 *
 * @todo Think about constness.
 */
template <class Iterator>
class Chunks {
public:
    typedef Iterator iterator_type;
    typedef boost::iterator_range<Iterator> Range;
    typedef typename std::iterator_traits<Iterator>::value_type value_type;
    typedef size_t size_type;
    typedef unsigned int number_type;
    typedef Chunk<iterator_type> chunk_type;
    typedef std::vector<chunk_type> chunks_type;
    typedef chunks_iterator<Chunks<iterator_type> > iterator;
public:
    Chunks();
    void appendChunk(const Iterator& first, const Iterator& last);
    void appendChunk(const Range& range);
    size_type size() const;
    number_type numberOfChunks() const;
    const chunk_type& chunk(number_type chunkNumber) const;
    const value_type& operator[](size_type position) const;
    iterator begin();
    iterator end();
private:
    chunks_type m_chunks;
    size_type m_size;
};


} // namespace MimeParser

template <class Iterator>
MimeParser::
Chunks<Iterator>::
Chunks()
    : m_size(0)
{}


template <class Iterator>
void
MimeParser::
Chunks<Iterator>::
appendChunk(const Iterator& first, const Iterator& last)
{
    appendChunk(boost::make_iterator_range(first, last));
}

template <class Iterator>
void
MimeParser::
Chunks<Iterator>::
appendChunk(const MimeParser::Chunks<Iterator>::Range& range)
{
    Chunk<Iterator> chunk(range);
    chunk.offset()=m_size;
    m_chunks.push_back(chunk);
    m_size+=chunk.range().size();
}

template <class Iterator>
typename MimeParser::Chunks<Iterator>::size_type
MimeParser::
Chunks<Iterator>::
size() const
{
    return m_size;
}

template <class Iterator>
const typename MimeParser::Chunks<Iterator>::value_type &
MimeParser::
Chunks<Iterator>::
operator[](MimeParser::Chunks<Iterator>::size_type position) const
{
    typename MimeParser::Chunks<Iterator>::chunks_type::const_iterator it;
    for (it=m_chunks.begin(); it!=m_chunks.end(); ++it) {
        if (it->offset()>position) {
            break;
        }
    }
    --it;
    return it->range()[position-it->offset()];
}

template <class Iterator>
typename MimeParser::Chunks<Iterator>::number_type
MimeParser::
Chunks<Iterator>::
numberOfChunks() const
{
    return static_cast<number_type>(m_chunks.size());
}

template <class Iterator>
const typename MimeParser::Chunks<Iterator>::chunk_type &
MimeParser::
Chunks<Iterator>::
chunk(MimeParser::Chunks<Iterator>::number_type chunkNumber) const
{
    return m_chunks[chunkNumber];
}

template <class Iterator>
typename MimeParser::Chunks<Iterator>::iterator
MimeParser::
Chunks<Iterator>::
begin()
{
    return iterator(this, 0, 0);
}

template <class Iterator>
typename MimeParser::Chunks<Iterator>::iterator
MimeParser::
Chunks<Iterator>::
end()
{
    return iterator(this, numberOfChunks(), 0);
}

#endif
