#ifndef EDP_EOL_PARSER_H_
#define EDP_EOL_PARSER_H_

#include <boost/range.hpp>

namespace MimeParser {
namespace Parsers {

/**
 * This Event-Driven parsers parses EOL events.
 *
 * @a Handler class must have endOfLine method,
 * and it will receives eol events from this parser.
 */
template <class Iterator, class Handler>
class EolParser {
public:
    typedef boost::iterator_range<Iterator> Range;
public:
    EolParser(const Iterator& first, Handler& handler);
    void push(const Iterator& last);
    void stop();
private:
    inline bool handleN();
    inline bool handleR(const Iterator& last);
private:
    Handler& m_handler;
    /**
     * Ok, this iterator in needed for real - it's parses an end-of-line events,
     * and we need think about this iterator while returning a most earlier
     * iterator in whole parser iterator range
     * #iterator
     */
    Iterator m_pos;
};

} // namespace Parsers
} // namespace MimeParser

template <class Iterator, class Handler>
MimeParser::
Parsers::
EolParser<Iterator, Handler>::
EolParser(const Iterator& first, Handler& handler)
    : m_handler(handler)
    , m_pos(first)
{}


template <class Iterator, class Handler>
void
MimeParser::
Parsers::
EolParser<Iterator, Handler>::
push(const Iterator& last)
{
    while (m_pos!=last) {
        switch (*m_pos) {
            case '\n' :
                if (!handleN()) {
                    return;
                }
                break;
            case '\r' :
                if (!handleR(last)) {
                    return;
                }
                break;
            default   :
                ++m_pos;
                break;
        }
    }
}

template <class Iterator, class Handler>
bool
MimeParser::
Parsers::
EolParser<Iterator, Handler>::
handleN()
{
    Iterator oldPos=m_pos;
    ++m_pos;
    return m_handler.endOfLine(Range(oldPos, m_pos));
}

template <class Iterator, class Handler>
bool
MimeParser::
Parsers::
EolParser<Iterator, Handler>::
handleR(const Iterator& last)
{
    if (m_pos+1==last) {
        return false;
    }
    if ('\n'==*(m_pos+1)) {
        Iterator oldPos=m_pos;
        m_pos+=2;
        return m_handler.endOfLine(Range(oldPos, m_pos));
    } else {
        ++m_pos;
    }
    return true;
}

template <class Iterator, class Handler>
void
MimeParser::
Parsers::
EolParser<Iterator, Handler>::
stop()
{
    /*NO-OP*/
}

#endif
