#ifndef EDP_HEADER_HANDLER_H_
#define EDP_HEADER_HANDLER_H_

#include <string>
#include <boost/range.hpp>
#include <boost/range/iterator_range.hpp>
#include <boost/algorithm/string/predicate.hpp>
#include <mimeparser/HeaderField.h>
#include <mimeparser/HeaderData.h>


namespace MimeParser {
namespace Handlers {

template <class ForwardTraversalIterator, class Data=HeaderData>
class HeaderHandler {
public:
    typedef enum {
        DEFAULT = 0,
        PARSE_FULL_MODE = 1
    } ParseMode;
public:
    typedef ForwardTraversalIterator Iterator;
    typedef boost::iterator_range<ForwardTraversalIterator> Range;
    typedef Data ParsedData;
public:
    HeaderHandler(ParseMode parseMode=DEFAULT, bool unfold = true)
        : m_isParsed(false)
        , m_parseMode(parseMode)
        , m_unfold(unfold)
        , m_data(unfold)
    {}
    bool beginHeader(const Iterator& /*it*/) {
        return true;
    }
    bool endHeader(const Iterator& /*it*/) {
        m_isParsed=true;
        return true;
    }
    bool headerField(const Range& range, const Range& eol);
    bool isParsed() const {
        return m_isParsed;
    }
    const Data& data() const {
        return m_data;
    }
private:
    bool m_isParsed;
    /// @note Maybe we need to implement it in functional way.
    ParseMode m_parseMode;
    bool m_unfold;
    Data m_data;
};

} // namespace Handlers
} // namespace MimeParser


template <class ForwardTraversalIterator, class Data>
bool
MimeParser::Handlers::
HeaderHandler<ForwardTraversalIterator, Data>::
headerField(const Range& range, const Range& eol)
{
    bool parsed = false;
    /// @note we can split that checks in functors.
    HeaderField headerField(range.begin(), eol.begin(), m_unfold);
    if (!headerField.isValid()) {
        return true;
    }
    const std::string& name=headerField.name();
    if (boost::istarts_with(name, "c")) {
        if (m_data.parseContentType(headerField)) {
            parsed = true;
        } else if (m_data.parseContentDisposition(headerField)) {
            parsed = true;
        } else if (m_data.parseContentTransferEncoding(headerField)) {
            parsed = true;
        } else if (m_data.parseContentId(headerField)) {
            parsed = true;
        }
    }
    if (PARSE_FULL_MODE & m_parseMode) {
        if (!parsed) {
            if (boost::iequals(name, "from")) {
                m_data.parseFrom(headerField);
            }
            if (boost::iequals(name, "to")) {
                m_data.parseTo(headerField);
            }
            if (boost::iequals(name, "cc")) {
                m_data.parseCc(headerField);
            }
            if (boost::iequals(name, "message-id")) {
                m_data.parseMessageId(headerField);
            }
        }
        m_data.addHeaderField(headerField);
    }
    return true;
}

#endif
