#ifndef EDP_HEADER_HANDLER_HELPER_H_
#define EDP_HEADER_HANDLER_HELPER_H_

#include <mimeparser/HeaderHandler.h>
#include <mimeparser/HeaderParser.h>

namespace MimeParser {

using Parsers::HeaderParser;
using Handlers::HeaderHandler;

/**
 * NO-OP header handler - used as example.
 */
template <class ForwardTraversalIterator>
class NoopHeaderHandler {
public:
    typedef ForwardTraversalIterator Iterator;
    typedef boost::iterator_range<Iterator> Range;
public:
    bool beginHeader(const Iterator& /*it*/) {
        return true;
    }
    bool endHeader(const Iterator& /*it*/) {
        return true;
    }
    bool headerField(const Range& /*range*/, const Range& /*eol*/) {
        return true;
    }
};


/**
 * Tee for header handler
 * This is a specific handler, that takes user handler, and uses
 * default handler in parsing, but also sends events to user handler.
 */

template
<class MainHandler,
class UserHandler=NoopHeaderHandler<
typename MainHandler::Iterator
>
>
class HeaderHandlerTee {
public:
    typedef typename MainHandler::Iterator Iterator;
    typedef typename MainHandler::ParsedData Data;
    typedef boost::iterator_range<Iterator> Range;
public:
    HeaderHandlerTee(MainHandler& mainHandler, UserHandler& userHandler)
        : m_mainHandler(mainHandler)
        , m_userHandler(userHandler)
    {}
    bool beginHeader(const Iterator& it) {
        bool resultMain=m_mainHandler.beginHeader(it);
        bool resultUser=m_userHandler.beginHeader(it);
        return resultMain && resultUser;
    }
    bool endHeader(const Iterator& it) {
        bool resultMain=m_mainHandler.endHeader(it);
        bool resultUser=m_userHandler.endHeader(it);
        return resultMain && resultUser;
    }
    bool headerField(const Range& range, const Range& eol) {
        bool resultMain=m_mainHandler.headerField(range, eol);
        bool resultUser=m_userHandler.headerField(range, eol);
        return resultMain && resultUser;
    }
    bool isParser() const {
        return m_mainHandler.isParsed();
    }
    const Data& data() const {
        return m_mainHandler.data();
    }
private:
    MainHandler& m_mainHandler;
    UserHandler& m_userHandler;
};

/**
 * This class helps user to provide his own handler and receive events
 */
template <class UserHandler>
class HeaderHandlerHelper {
public:
    typedef typename UserHandler::Iterator Iterator;
    typedef HeaderHandler<Iterator> MainHandler;
    typedef HeaderHandlerTee<MainHandler, UserHandler> TeeHandler;
    typedef boost::iterator_range<Iterator> Range;
    typedef HeaderParser<Iterator, TeeHandler> Parser;
public:
    HeaderHandlerHelper(UserHandler& userHandler)
        : m_userHandler(userHandler)
        , m_mainHandler()
        , m_teeHandler(m_mainHandler, m_userHandler)
    {}
    TeeHandler& handler() {
        return m_teeHandler;
    }
private:
    UserHandler& m_userHandler;
    MainHandler m_mainHandler;
    TeeHandler m_teeHandler;
};

/**
 * Simple all-in-one function to just parse header.
 */
template<class UserHandler>
void
parse_header(UserHandler& userHandler,
             const typename UserHandler::Iterator& begin,
             const typename UserHandler::Iterator& end)
{
    typedef HeaderHandlerHelper<UserHandler> Helper;
    typedef typename Helper::Parser Parser;
    Helper helper(userHandler);
    Parser parser(begin, helper.handler());
    parser.push(end);
    parser.stop();
}
};

#endif
