#ifndef EDP_MESSAGE_HANDLER_H_
#define EDP_MESSAGE_HANDLER_H_

#include <mimeparser/MessageStructure.h>
#include <mimeparser/XmlPartInfo.h>
#include <mimeparser/Hid.h>
#include <mimeparser/HeaderHandler.h>
#include <mimeparser/BoundaryStatus.h>
#include <mimeparser/BodyLength.h>

#include <string>
#include <stack>
#include <vector>

namespace MimeParser {
namespace Handlers {

/**
 * This class have to be splitted with tee handlers.
 */
template <
class ForwardTraversalIterator,
      class HHandler=HeaderHandler<ForwardTraversalIterator>
      >
class MessageHandler {
public:
    typedef ForwardTraversalIterator Iterator;
    typedef HHandler HeaderHandler;
    typedef boost::iterator_range<Iterator> Range;
    typedef BodyLength::Calculator<Iterator> Calculator;
    typedef BodyLength::Plain<Iterator> PlainCalculator;
    typedef BodyLength::Qp<Iterator> QpCalculator;
    typedef BodyLength::Base64<Iterator> Base64Calculator;
    typedef std::string ErrorEntry;
    typedef std::vector<ErrorEntry> ErrorList;
public:
    MessageHandler();

    bool beginMessage(const Iterator& position);
    bool endMessage(const Iterator& position);

    bool beginPart(const Iterator& position, const HHandler& hHandler);
    bool endPart(const Iterator& position);

    bool beginHeader(const Iterator& position);
    bool endHeader(const Iterator& position);

    bool handleBodyLine(const Range& range);
public:
    const MessageStructure& result() const {
        return m_messageStructure;
    }

    void addError(const ErrorEntry& error);
    void getErrors(ErrorList& errorList);
private:
    static bool isContainerPart(const XmlPartInfo& xmlPartInfo);
private:
    /**
     * #iterator
     * Here we have a stack of iterators, and we need to get rid of that.
     * I propose: we must add in all callbacks (like beginPart, etc)
     * not only iterator, but an a offset (maybe inside iterator), and here we
     * just can store offsets, not iterators for length calculation
     */
    std::stack<Iterator> m_offsets;
    MessageStructure m_messageStructure;
    PartTraverser m_partTraverser;
    /**
     * Must remove those iterator - we are using it only for lenght calculation
     * #iterator
     */
    Iterator m_start;
    std::unique_ptr<Calculator> m_calculator;

    std::vector<std::string> errorList_;
};

} // namespace Handlers
} // namespace MimeParser

template <class ForwardTraversalIterator, class HHandler>
MimeParser::Handlers::
MessageHandler<ForwardTraversalIterator, HHandler>::
MessageHandler()
    : m_partTraverser(NULL)
{}

template <class ForwardTraversalIterator, class HHandler>
bool
MimeParser::Handlers::
MessageHandler<ForwardTraversalIterator, HHandler>::
beginMessage(const Iterator& position)
{
    m_start=position;
    return true;
}

template <class ForwardTraversalIterator, class HHandler>
bool
MimeParser::Handlers::
MessageHandler<ForwardTraversalIterator, HHandler>::
endMessage(const Iterator& position)
{
    if (!m_messageStructure.hasStructure()) {
        std::unique_ptr<PartStructure> partStructure(new PartStructure());
        partStructure->info().setOffset(position-m_start);
        // FIXME take from default part
        partStructure->info().setType("text");
        partStructure->info().setSubtype("plain");
        partStructure->info().setCharset("US-ASCII");
        partStructure->info().setContentTransferEncoding("7bit");
        m_messageStructure.setPartStructure(partStructure.release());
    }
    return true;
}

template <class ForwardTraversalIterator, class HHandler>
bool
MimeParser::Handlers::
MessageHandler<ForwardTraversalIterator, HHandler>::
beginPart(const Iterator& position, const HHandler& hHandler)
{
    m_offsets.push(position);
    if (!m_messageStructure.hasStructure()) {
        m_messageStructure.setPartStructure(new PartStructure());
        m_partTraverser=PartTraverser(m_messageStructure.partStructure());
    } else {
        if (isContainerPart((*m_partTraverser).info())) {
            PartStructure* p=&((*m_partTraverser).newChild());
            m_partTraverser=PartTraverser(p);
        } else {
            m_partTraverser.up();
            PartStructure& pstruct=(*m_partTraverser);
            PartStructure* p=&(pstruct.newChild());
            m_partTraverser=PartTraverser(p);
        }
    }
    const HeaderData& headerData=hHandler.data();
    XmlPartInfo& info=(*m_partTraverser).info();
    info.setOffset(position-m_start);
    info.setType(headerData.cType());
    info.setSubtype(headerData.contentSubtype());
    info.setCharset(headerData.charset());
    info.setName(headerData.name());
    info.setContentDisposition(headerData.content_disposition());
    info.setFilename(headerData.filename());
    info.setContentTransferEncoding(headerData.content_transfer_encoding());
    info.setContentId(headerData.content_id());
    if ("multipart"!=info.type()) {
        const std::string& cte=info.contentTransferEncoding();
        if ("quoted-printable"==cte) {
            m_calculator.reset(new QpCalculator());
        } else if ("base64"==cte) {
            m_calculator.reset(new Base64Calculator());
        } else {
            m_calculator.reset(new PlainCalculator());
        }
    }
    return true;
}

template <class ForwardTraversalIterator, class HHandler>
bool
MimeParser::Handlers::
MessageHandler<ForwardTraversalIterator, HHandler>::
endPart(const Iterator& position)
{
    if (NULL!=m_calculator.get()) {
        XmlPartInfo& info=(*m_partTraverser).info();
        info.setRealLength(m_calculator->length());
        m_calculator.reset();
    }
    if (m_offsets.empty()) {
        return true;
    }
    (*m_partTraverser).info().setLength(position-m_offsets.top());
    m_offsets.pop();
    m_partTraverser.up();
    return true;
}

template <class ForwardTraversalIterator, class HHandler>
bool
MimeParser::Handlers::
MessageHandler<ForwardTraversalIterator, HHandler>::
beginHeader(const Iterator& /*position*/)
{
    return true;
}

template <class ForwardTraversalIterator, class HHandler>
bool
MimeParser::Handlers::
MessageHandler<ForwardTraversalIterator, HHandler>::
endHeader(const Iterator& /*position*/)
{
    return true;
}
#include <iostream>
template <class ForwardTraversalIterator, class HHandler>
bool
MimeParser::Handlers::
MessageHandler<ForwardTraversalIterator, HHandler>::
handleBodyLine(const Range& range)
{
    if (NULL!=m_calculator.get()) {
        m_calculator->push(range.begin(), range.end());
    }
    return true;
}

template <class ForwardTraversalIterator, class HHandler>
bool
MimeParser::Handlers::
MessageHandler<ForwardTraversalIterator, HHandler>::
isContainerPart(const XmlPartInfo& xmlPartInfo)
{
    if (boost::iequals("message", xmlPartInfo.type()) &&
            boost::iequals("rfc822", xmlPartInfo.subtype())) {
        return true;
    }
    if (boost::iequals("multipart", xmlPartInfo.type())) {
        return true;
    }
    return 1==xmlPartInfo.hid().depth();
}

template <class ForwardTraversalIterator, class HHandler>
void
MimeParser::Handlers::
MessageHandler<ForwardTraversalIterator, HHandler>::
addError(const ErrorEntry& error)
{
    std::string errorPrefix = "No part";
    if ( m_messageStructure.hasStructure() ) {
        if(m_messageStructure.partStructure()->size() != 0) {
            size_t partNumber = m_messageStructure.partStructure()->size()-1;
            errorPrefix = "Part #" + m_messageStructure.partStructure()->getChild(partNumber).info().hid().toString();
        }
    }
    errorList_.push_back(errorPrefix + ": " + error);
}

template <class ForwardTraversalIterator, class HHandler>
void
MimeParser::Handlers::
MessageHandler<ForwardTraversalIterator, HHandler>::
getErrors(ErrorList& errorList)
{
    errorList = errorList_;
}

#endif
