#pragma once

#include <mimeparser/MessageHandler.h>
#include <mimeparser/MessageParser.h>

namespace MimeParser {

using namespace Handlers;
using namespace Parsers;

template <
class ForwardTraversalIterator,
      class HHandler=HeaderHandler<ForwardTraversalIterator>
      >
class NoopMessageHandler {
public:
    typedef ForwardTraversalIterator Iterator;
public:
    bool beginMessage(const Iterator& /*position*/) {
        return true;
    }
    bool endMessage(const Iterator& /*position*/) {
        return true;
    }
    bool beginPart(const Iterator& /*position*/,
                   const HHandler& /*hHandler*/) {
        return true;
    }
    bool endPart(const Iterator& /*position*/) {
        return true;
    }
    bool beginHeader(const Iterator& /*position*/) {
        return true;
    }
    bool endHeader(const Iterator& /*position*/) {
        return true;
    }
};

template <
class MainHandler,
      class UserHandler=MessageHandler
      <
      typename MainHandler::Iterator,
      typename MainHandler::HeaderHandler
      >
      >
class MessageHandlerTee {
public:
    typedef typename MainHandler::Iterator Iterator;
    typedef typename MainHandler::HeaderHandler HHandler;
    typedef boost::iterator_range<Iterator> Range;
public:
    MessageHandlerTee(MainHandler& mainHandler,
                      UserHandler& userHandler)
        : m_mainHandler(mainHandler)
        , m_userHandler(userHandler)
    {}
    bool beginMessage(const Iterator& position) {
        bool mainResult=m_mainHandler.beginMessage(position);
        bool userResult=m_userHandler.beginMessage(position);
        return mainResult && userResult;
    }
    bool endMessage(const Iterator& position) {
        bool mainResult=m_mainHandler.endMessage(position);
        bool userResult=m_userHandler.endMessage(position);
        return mainResult && userResult;
    }
    bool beginPart(const Iterator& position, const HHandler& hHandler) {
        bool mainResult=m_mainHandler.beginPart(position, hHandler);
        bool userResult=m_userHandler.beginPart(position, hHandler);
        return mainResult && userResult;
    }
    bool endPart(const Iterator& position) {
        bool mainResult=m_mainHandler.endPart(position);
        bool userResult=m_userHandler.endPart(position);
        return mainResult && userResult;
    }
    bool beginHeader(const Iterator& position) {
        bool mainResult=m_mainHandler.beginHeader(position);
        bool userResult=m_userHandler.beginHeader(position);
        return mainResult && userResult;
    }
    bool endHeader(const Iterator& position) {
        bool mainResult=m_mainHandler.endHeader(position);
        bool userResult=m_userHandler.endHeader(position);
        return mainResult && userResult;
    }
    bool handleBodyLine(const Range& range) {
        return m_mainHandler.handleBodyLine(range);
    }
    void addError(const std::string&) {
    }
private:
    MainHandler& m_mainHandler;
    UserHandler& m_userHandler;
};

template <class UserHandler>
class MessageHandlerHelper {
public:
    typedef typename UserHandler::Iterator Iterator;
    typedef HeaderHandler<Iterator> HHandler;
    typedef MessageHandler<Iterator, HHandler> MainHandler;
    typedef MessageHandlerTee<MainHandler, UserHandler> TeeHandler;
    typedef MessageParser<Iterator, TeeHandler> Parser;
public:
    MessageHandlerHelper(UserHandler& userHandler)
        : m_userHandler(userHandler)
        , m_mainHandler()
        , m_teeHandler(m_mainHandler, m_userHandler)
    {}
    TeeHandler& handler() {
        return m_teeHandler;
    }
private:
    UserHandler& m_userHandler;
    MainHandler m_mainHandler;
    TeeHandler m_teeHandler;
};

template <class UserHandler>
void
parse_message(UserHandler& userHandler,
              const typename UserHandler::Iterator& begin,
              const typename UserHandler::Iterator& end)
{
    typedef MessageHandlerHelper<UserHandler> Helper;
    typedef typename Helper::Parser Parser;
    Helper helper(userHandler);
    Parser parser(begin, helper.handler());
    parser.push(end);
    parser.stop();
}
}
