#pragma once

#include <mimeparser/MessageHandler.h>
#include <mimeparser/HeaderParser.h>


#include <boost/algorithm/string/predicate.hpp>
#include <boost/algorithm/string/find.hpp>

#include <vector>
#include <memory>
#include <stdexcept>

namespace MimeParser {
namespace Parsers {

using Handlers::MessageHandler;
using Handlers::HeaderHandler;
using Parsers::HeaderParser;

template <class ForwardTraversalIterator>
class HeaderHandlerFactory {
public:
    typedef ForwardTraversalIterator Iterator;
    typedef HeaderHandler<Iterator> Handler;
public:
    HeaderHandlerFactory()
        :   m_doParseInline(false)
        ,   m_doCalculateRealLength(false)
    {}
    std::unique_ptr<Handler> create() {
        return std::unique_ptr<Handler>(new Handler());
    }
    void setParseInline(bool doParseInline) {
        m_doParseInline=doParseInline;
    }
    /// @warning FIXME should be a param to parser
    bool doParseInline() const {
        return m_doParseInline;
    }
    void setCalculateRealLength(bool doCalculateRealLength) {
        m_doCalculateRealLength=doCalculateRealLength;
    }
    bool doCalculateRealLength() const {
        return m_doCalculateRealLength;
    }
private:
    bool m_doParseInline;
    bool m_doCalculateRealLength;
};

template <
class Iterator,
      class Handler=MessageHandler<Iterator>,
      class Factory=HeaderHandlerFactory<Iterator>
      >
class MessageParser {
public:
    typedef boost::iterator_range<Iterator> Range;
    typedef typename Factory::Handler MyHeaderHandler;
    typedef HeaderParser<Iterator, MyHeaderHandler> MyHeaderParser;
    typedef EolParser<Iterator, MessageParser> MyEolParser;
    typedef boost::iterator_range<std::string::const_iterator> StringRange;
    typedef std::vector<MyHeaderHandler*> PartStack;
    /**
     * This is per-line state.
     * (Line , State) --> (Line+1, Parser(State, Line))
     */
    typedef enum {
        /// Parser is in this state while header is parsed
        PARSING_HEADER,
        /// Parser is in this state while searching for boundary
        FINDING_BOUNDARY,
        /// Parser is in this state while body is parsed
        PARSING_BODY
    } State;
public:
    /**
     * Creates parser
     * @note factory is passed by value.
     */
    MessageParser(const Iterator& first,
                  Handler& handler,
                  Factory factory=Factory());
    ~MessageParser();
    void push(const Iterator& last);
    void stop();
public:
    bool endOfLine(const Range& range);
private:
    void eolHandler(const Range& range);
    void handleParsingHeaderState(const Range& range);
    void handleFindingBoundaryState(const Range& range);
    void handleParsingBodyState(const Range& range);
    inline void sendBodyLine(const Range& range);
    inline void setState(State state);
    inline Range currentLine(const Range& range) const;
    inline BoundaryStatus checkBoundary(const Range& range);
    inline BoundaryStatus verifyBoundary(const Range& range);
private:
    /// MessageHandler -receives all events from parser
    Handler& m_handler;
    /// Factory of HeaderHandler's
    Factory m_factory;
    /**
     * #iterator
     * This is an last iterator - nothing bad that we are storing it
     */
    Iterator m_last;
    /**
     * #iterator
     * Need to get rid of that and store only offset
     */
    Iterator m_partStart;
    /**
     * #iterator
     * Need to get rid of that and store only offset
     */
    Iterator m_bodyStart;
    /**
     * #iterator
     * Iterator of last end-of-line, we need it
     */
    Range m_lastEol;
    /**
     * #iterator
     * @todo FIXME we need to use the same EolParser, got exatly the same in
     * HeaderParsed, so it will be nice, that we will reuse it in HeaderParser
     */
    MyEolParser m_eolParser;
    State m_state;
    /// #iterator inside
    std::unique_ptr<MyHeaderHandler> m_headerHandler;
    /// #iterator inside
    std::unique_ptr<MyHeaderParser> m_headerParser;
    /**
     * #iterator
     * Must get rid of that, we need only offsets here and @a HeaderData
     */
    PartStack m_parts;
    bool m_parseInlineMessage;
private:
    static inline bool validMessageContentTransferEncoding(const std::string& contentTransferEncoding);
};

} // namespace Parsers
} // namespace MimeParser

template <class Iterator, class Handler, class Factory>
MimeParser::Parsers::
MessageParser<Iterator, Handler, Factory>::
MessageParser(const Iterator& first, Handler& handler, Factory factory)
    : m_handler(handler)
    , m_factory(factory)
    , m_last(first)
    , m_partStart(first)
    , m_bodyStart(first)
    , m_lastEol(first, first)
    , m_eolParser(first, *this)
    , m_state(PARSING_HEADER)
    , m_parseInlineMessage(m_factory.doParseInline())
{
    m_handler.beginMessage(first);
}

template <class Iterator, class Handler, class Factory>
MimeParser::Parsers::
MessageParser<Iterator, Handler, Factory>::
~MessageParser()
{
    while (!m_parts.empty()) {
        delete m_parts.back();
        m_parts.pop_back();
    }
}

template <class Iterator, class Handler, class Factory>
void
MimeParser::Parsers::
MessageParser<Iterator, Handler, Factory>::
push(const Iterator& last)
{
    m_last=last;
    m_eolParser.push(last);
}

template <class Iterator, class Handler, class Factory>
void
MimeParser::Parsers::
MessageParser<Iterator, Handler, Factory>::
stop()
{
    // here we must report all another things
    while (!m_parts.empty()) {
        m_handler.endPart(m_last);
        delete m_parts.back();
        m_parts.pop_back();
    }
    m_handler.endPart(m_last);
    m_handler.endMessage(m_last);
}

template <class Iterator, class Handler, class Factory>
bool
MimeParser::Parsers::
MessageParser<Iterator, Handler, Factory>::
endOfLine(const Range& range)
{
    eolHandler(range);
    m_lastEol=range;
    return true;
}

template <class Iterator, class Handler, class Factory>
void
MimeParser::Parsers::
MessageParser<Iterator, Handler, Factory>::
eolHandler(const Range& range)
{
    switch (m_state) {
        case PARSING_HEADER: {
            handleParsingHeaderState(range);
        }
        break;
        case FINDING_BOUNDARY: {
            handleFindingBoundaryState(range);
        }
        break;
        case PARSING_BODY: {
            handleParsingBodyState(range);
        }
        break;
    }
}

template <class Iterator, class Handler, class Factory>
void
MimeParser::Parsers::
MessageParser<Iterator, Handler, Factory>::
handleParsingHeaderState(const Range& range)
{
    if (NULL==m_headerParser.get()) {
        m_headerHandler=m_factory.create();
        m_headerParser.reset(
            new MyHeaderParser(m_partStart, *m_headerHandler));
        m_handler.beginHeader(range.begin());
    }
    m_headerParser->push(range.end());
    if (m_headerHandler->isParsed()) {
        m_bodyStart=range.end();
        m_handler.endHeader(m_bodyStart);
        m_handler.beginPart(m_bodyStart, *m_headerHandler);
        m_headerParser.reset();
        const auto& headerData=m_headerHandler->data();
        if (boost::iequals("multipart", headerData.cType())) {
            if (!validMessageContentTransferEncoding(headerData.content_transfer_encoding())) {
                m_handler.addError("Multipart message Transfer-Content-Type field is incorrect");
            }

            if (!headerData.boundary().empty()) {
                m_parts.push_back(m_headerHandler.release());
                setState(FINDING_BOUNDARY);
            } else {
                m_handler.addError("Empty boundary");
                setState(PARSING_BODY);
            }
        } else if (headerData.isInlineMessage()) {
            if (m_parseInlineMessage) {
                 if (!validMessageContentTransferEncoding(headerData.content_transfer_encoding())) {
                     m_handler.addError("Inline message Transfer-Content-Type field is incorrect");
                 }

                 ///@todo FIXME nested bug, example: 4088.0.1275452663162487721261946155060
                 m_partStart=range.end();
                 m_parts.push_back(m_headerHandler.release());
                 setState(PARSING_HEADER);
            } else {
                setState(PARSING_BODY);
            }
        } else {
            setState(PARSING_BODY);
        }
    }
}

template <class Iterator, class Handler, class Factory>
void
MimeParser::Parsers::
MessageParser<Iterator, Handler, Factory>::
handleFindingBoundaryState(const Range& range)
{
    BoundaryStatus lineBoundaryStatus=checkBoundary(range);
    if (BoundaryStatus::BOUNDARY==lineBoundaryStatus.status()) {
        // doesn't matter where the boundary is, we'll go next line
        m_partStart=range.end();
        setState(PARSING_HEADER);
    } else if (BoundaryStatus::FINAL_BOUNDARY==lineBoundaryStatus.status()) {
        // wtf? it shouldn't be.
    }
}

template <class Iterator, class Handler, class Factory>
void
MimeParser::Parsers::
MessageParser<Iterator, Handler, Factory>::
sendBodyLine(const Range& range)
{
    if (m_factory.doCalculateRealLength()) {
        m_handler.handleBodyLine(range);
    }
}

template <class Iterator, class Handler, class Factory>
void
MimeParser::Parsers::
MessageParser<Iterator, Handler, Factory>::
handleParsingBodyState(const Range& range)
{
    if (m_parts.empty()) {
        sendBodyLine(Range(m_lastEol.end(), range.end()));
        return;// switch and break;
    }
    BoundaryStatus lineBoundaryStatus=checkBoundary(range);
    if (BoundaryStatus::NO_BOUNDARY==lineBoundaryStatus.status()) {
        sendBodyLine(Range(m_lastEol.end(), range.end()));
        return;
    }
    Iterator partEnd=m_lastEol.begin()+lineBoundaryStatus.offset();
    partEnd=(m_bodyStart>partEnd)?m_bodyStart:partEnd;
    sendBodyLine(Range(m_lastEol.begin(), partEnd));
    if (lineBoundaryStatus.nestedLevel()>0) {
        ///@todo FIXME working not properly for inline messages
        //std::cerr << "NESTED BUG!" << std::endl;
        for (size_t i=0; i<lineBoundaryStatus.nestedLevel(); ++i) {
            delete m_parts.back();
            m_parts.pop_back();
            m_handler.endPart(partEnd);
        }
    }
    if (BoundaryStatus::BOUNDARY==lineBoundaryStatus.status()) {
        m_partStart=range.end();
        m_handler.endPart(partEnd);
        setState(PARSING_HEADER);
    } else if (BoundaryStatus::FINAL_BOUNDARY==lineBoundaryStatus.status()) {
        delete m_parts.back();
        m_parts.pop_back();
        m_handler.endPart(partEnd);
    }
}

template <class Iterator, class Handler, class Factory>
void
MimeParser::Parsers::
MessageParser<Iterator, Handler, Factory>::
setState(State state)
{
    /*
      std::cerr << "State: " << m_state << "->" << state << std::endl;
      std::cerr << m_parts.size() << std::endl;
    */
    m_state=state;
}

template <class Iterator, class Handler, class Factory>
typename MimeParser::Parsers::MessageParser<Iterator, Handler, Factory>::
Range
MimeParser::Parsers::
MessageParser<Iterator, Handler, Factory>::
currentLine(const Range& range) const
{
    return boost::make_iterator_range(m_lastEol.end(), range.begin());
}

template <class Iterator, class Handler, class Factory>
MimeParser::BoundaryStatus
MimeParser::Parsers::
MessageParser<Iterator, Handler, Factory>::
checkBoundary(const Range& range)
{
    Range line=boost::make_iterator_range(m_lastEol.end(), range.begin());
    /// @warning we can use boost::starts_with here due to rfc
    Range bstart=boost::find_first(line, BoundaryStatus::delim());
    if (!bstart.empty()) {
        size_t offset=((bstart.begin()-line.begin())>0)?(bstart.begin()-m_lastEol.begin()):0;
        return verifyBoundary(boost::make_iterator_range(bstart.begin(), line.end())).setOffset(offset);
    }
    return BoundaryStatus(BoundaryStatus::NO_BOUNDARY);
}

#include <iostream>

template <class Iterator, class Handler, class Factory>
MimeParser::BoundaryStatus
MimeParser::Parsers::
MessageParser<Iterator, Handler, Factory>::
verifyBoundary(const Range& fbline)
{
//    Range bstart=boost::make_iterator_range(fbline.begin(), fbline.begin()+BoundaryStatus::delim().size()); // FIXME
    Range bline=boost::make_iterator_range(fbline.begin()+BoundaryStatus::delim().size(), fbline.end()); // FIXME
    typedef typename PartStack::const_reverse_iterator RPartIterator;
    size_t partNum=0;
    for (RPartIterator it=m_parts.rbegin(); it!=m_parts.rend(); ++it) {
        MyHeaderHandler& myHeaderHandler=*(*it);
        const auto& headerData=myHeaderHandler.data();
        StringRange boundary=boost::make_iterator_range(headerData.boundary().begin(), headerData.boundary().end());
        if (!headerData.isInlineMessage() && boost::starts_with(bline, boundary)) {
            //std::cerr << partNum << "/" << m_parts.size() << std::endl;
            //std::cerr << boundary << std::endl;
            //std::cerr << fbline << std::endl;
            Range brange=boost::make_iterator_range(bline.begin(), bline.begin()+boundary.size());
            Range after_boundary=boost::make_iterator_range(brange.end(), fbline.end());
            if (boost::starts_with(after_boundary, BoundaryStatus::delim())) {
                return BoundaryStatus(BoundaryStatus::FINAL_BOUNDARY, partNum);
            }
            return BoundaryStatus(BoundaryStatus::BOUNDARY, partNum);
        }
        ++partNum;
    }
    return BoundaryStatus(BoundaryStatus::NO_BOUNDARY);
}

template <class Iterator, class Handler, class Factory>
inline bool
MimeParser::Parsers::
MessageParser<Iterator, Handler, Factory>::
validMessageContentTransferEncoding(const std::string& contentTransferEncoding)
{
    return boost::iequals("7bit", contentTransferEncoding) ||
           boost::iequals("8bit", contentTransferEncoding) ||
           boost::iequals("binary", contentTransferEncoding);
}
