#ifndef BUTIL_BASE64
#define BUTIL_BASE64

#include <cstddef>
#include <boost/range/iterator_range.hpp>

/**
 * @file
 *
 * @todo add cc
 *
 * Here some links and implementation related to base64 encoding
 * http://en.wikipedia.org/wiki/Base64
 */

namespace mail { namespace utils { namespace base64 {

    class traits {
        public:
            static size_t reminderOfEquals(size_t equalsSigns) {
                if (2==equalsSigns) {
                    return 1;
                } else if (1==equalsSigns) {
                    return 2;
                }
                return 0;
            }
            static const char EQUALS_SIGN='=';
            static inline bool isBase64Char(char c) {
                if (('a'<=c && 'z'>=c) ||
                    ('A'<=c && 'Z'>=c) ||
                    ('0'<=c && '9'>=c) ||
                    ('+'==c) ||
                    ('/'==c)) {
                        return true;
                    } else {
                        return false;
                    }
                }
    };
    /**
     *  Calculate length of right-encoded in base64.
     */
    template <class Iterator>
    size_t calculate_length(const boost::iterator_range<Iterator> range) {
        size_t result=0;
        unsigned int equalsSigns=0;
        unsigned short quarter=0;
        Iterator it=range.begin();
        while (it!=range.end()) {
            if (traits::isBase64Char(*it)) {
                // if (0!=equalsSigns) {/*throw error*/}
                ++quarter;
                if (4==quarter) {
                    result+=3;
                    quarter=0;
                }
            } else if (traits::EQUALS_SIGN==*it) {
                ++equalsSigns;
            }
            ++it;
        }
        result+=traits::reminderOfEquals(equalsSigns);
        return result;
    }

    /**
     * Stream-based base64 length calculator
     */
    template <class ForwardTraversalIterator>
    class LengthCalculator {
        private:
            typedef enum {
                PARSING_BASE64,
                PARSING_SUFFIX
            } State;
        public:
            typedef ForwardTraversalIterator Iterator;
            typedef boost::iterator_range<Iterator> Range;
        public:
            typedef enum {
                NOTHING,
                // not implemented, thout that will be checking on validity
                CHECK_SYMBOLS
            } Flags;
        public:
            LengthCalculator(Flags flags=NOTHING)
            : m_flags(flags)
            , m_state(PARSING_BASE64)
            , m_isCorrect(true)
            , m_length(0)
            , m_b64length(0)
            , m_b64reminder(0)
            {}
            void push(const Range& range) {
                return push(range.begin(), range.end());
            }
            void push(const Iterator& begin, const Iterator& end) {
                for (Iterator it=begin; it!=end; ++it) {
                    processChar(*it);
                }
                m_length+=3*(m_b64length/4);
                m_b64length=m_b64length%4;
            }
            void stop() {
                m_length+=3*(m_b64length/4);
                m_length+=traits::reminderOfEquals(m_b64reminder);
            }
            size_t length() const {
                return m_length;
            }
            /**
             * Not implemented
             */
            bool isCorrect() const {
                return m_isCorrect;
            }
        private:
            inline void processChar(char c) {
                switch (m_state) {
                    case PARSING_BASE64 : {
                        if (traits::isBase64Char(c)) {
                            processBase64Char();
                        } else if (traits::EQUALS_SIGN==c) {
                            m_state=PARSING_SUFFIX;
                            processEqualsSign();
                        }
                    } break;
                    case PARSING_SUFFIX : {
                        if (traits::EQUALS_SIGN==c) {
                            processEqualsSign();
                        }
                    } break;
                }
            }
            inline void processBase64Char() {
                ++m_b64length;
            }
            inline void processEqualsSign() {
                ++m_b64reminder;
            }
        private:
            Flags m_flags;
            State m_state;
            bool m_isCorrect;
            size_t m_length;
            size_t m_b64length;
            size_t m_b64reminder;
    };


} // namespace qp
} // namespace utils
} // namespace mail


#endif
