#ifndef EDP_CHUNKS_ITERATOR_H_
#define EDP_CHUNKS_ITERATOR_H_

#include <iterator>

#include <boost/iterator/iterator_facade.hpp>

namespace MimeParser {

/**
 * This iterator helps iterate through collections of ranges
 * Implementation of this collection is inside @a ChunksType
 * We can iterate only straight forward to enable ChunksType freeing
 * @todo implement this feature
 */
template <class ChunksType>
class chunks_iterator : public boost::iterator_facade
    <
    chunks_iterator<ChunksType>,
    typename std::iterator_traits<typename ChunksType::iterator_type>::value_type,
    boost::random_access_traversal_tag
        > {
public:
    friend class boost::iterator_core_access;
    typedef typename std::iterator_traits<typename ChunksType::iterator_type>::value_type base_type;
public:
    explicit
    chunks_iterator(
        ChunksType* chunks=NULL,
        typename ChunksType::number_type chunkNumber=0,
        typename ChunksType::size_type relativeShift=0
    );
private:
    void increment();
    bool equal(chunks_iterator const& other) const;
    base_type& dereference() const;
    int distance_to(chunks_iterator const& other) const;
    void advance(size_t);
private:
    size_t calculateAbsoluteShift();
private:
    /**
     * implementation of chunks storage
     */
    ChunksType* m_chunks;
    typename ChunksType::number_type m_chunkNumber;
    typename ChunksType::size_type m_relativeShift;
    typename ChunksType::size_type m_absoluteShift;
};
} // namespace MimeParser


template <class ChunksType>
MimeParser::
chunks_iterator<ChunksType>::
chunks_iterator(
    ChunksType* chunks,
    typename ChunksType::number_type chunkNumber,
    typename ChunksType::size_type relativeShift
)
    : m_chunks(chunks)
    , m_chunkNumber(chunkNumber)
    , m_relativeShift(relativeShift)
{
    m_absoluteShift=calculateAbsoluteShift();
}

template <class ChunksType>
void
MimeParser::
chunks_iterator<ChunksType>::
increment()
{
    ++m_absoluteShift;
    if ((m_relativeShift+1)<m_chunks->chunk(m_chunkNumber).size()) {
        ++m_relativeShift;
    } else {
        m_relativeShift=0;
        ++m_chunkNumber;
    }
}

template <class ChunksType>
bool
MimeParser::
chunks_iterator<ChunksType>::
equal(const MimeParser::chunks_iterator<ChunksType>& other) const
{
    return (
               m_chunks==other.m_chunks &&
               //m_absoluteShift==other.m_absoluteShift &&
               m_relativeShift==other.m_relativeShift &&
               m_chunkNumber==other.m_chunkNumber
           );
}

template <class ChunksType>
typename MimeParser::chunks_iterator<ChunksType>::base_type&
MimeParser::
chunks_iterator<ChunksType>::
dereference() const
{
    // FIXME
    return const_cast<char&>(m_chunks->chunk(m_chunkNumber).range()[m_relativeShift]);
}

template <class ChunksType>
int
MimeParser::
chunks_iterator<ChunksType>::
distance_to(const MimeParser::chunks_iterator<ChunksType>& other) const
{
    return static_cast<int>(other.m_absoluteShift)-
           static_cast<int>(m_absoluteShift);
}

template <class ChunksType>
void
MimeParser::
chunks_iterator<ChunksType>::
advance(size_t n)
{
    size_t relativeAdvance=m_relativeShift+n;
    m_absoluteShift+=relativeAdvance;
    m_relativeShift=0;
    while (relativeAdvance>=m_chunks->chunk(m_chunkNumber).size()) {
        relativeAdvance-=m_chunks->chunk(m_chunkNumber).size();
        ++m_chunkNumber;
    }
    m_relativeShift=relativeAdvance;
}

template <class ChunksType>
size_t
MimeParser::
chunks_iterator<ChunksType>::
calculateAbsoluteShift()
{
    size_t result=0;
    for (typename ChunksType::number_type i=0; i<m_chunkNumber; ++i) {
        result+=m_chunks->chunk(i).size();
    }
    result+=m_relativeShift;
    return result;
}

#endif
