/*! @file libxml_helpers.h
 *  Содержит интересные определения типов, например для libxml-модуля xpath и структуры xmlPtr.
 */

#ifndef YANDEX_COMMON_LIBXML_HELPERS_H
#define YANDEX_COMMON_LIBXML_HELPERS_H

#include "libxml/globals.h"
#include "libxml/xpath.h"
#include "libxml/uri.h"

namespace libxml {

template<typename T>
struct xml_traits {
};

template<typename T>
/// Обладает семантикой "умного" указателя, куда передается функция, которая удаляет сам этот указатель.
/*! Также есть несколько интересных определений типов:
 *  - ::CharPtr
 *  - ::DocPtr
 *  - ::URIPtr
 *  - ::NodePtr
 *  - ::SignedCharPtr
 *  - ::OutputBufferPtr
 *  - ::ParserCtxtPtr
 *  - ::XPathContextPtr
 *  - ::XPathObjectPtr
 */
struct xmlPtr {
    T data_;

    typedef xml_traits<T> traits;

    xmlPtr(T t) : data_(t) { };
    ~xmlPtr() {
        if (data_) {
            traits::delete_data(data_);
        }
    };

    operator T() {
        return data_;
    };
    T get() const {
        return data_;
    };
    T release() {
        T tmp = data_;
        data_ = 0;
        return tmp;
    }
    T operator->() {
        return data_;
    };

private:
    xmlPtr(const xmlPtr<T>&);
    xmlPtr& operator=(const xmlPtr<T>&);
};

template<>
struct xml_traits<char*> {
    static void delete_data(char* d) {
        xmlFree(d);
    };
};

template<>
struct xml_traits<xmlChar*> {
    static void delete_data(xmlChar* d) {
        xmlFree(d);
    };
};

template<>
struct xml_traits<xmlOutputBufferPtr> {
    static void delete_data(xmlOutputBufferPtr b) {
        xmlOutputBufferClose(b);
    };
};

template<>
struct xml_traits<xmlXPathContextPtr> {
    static void delete_data(xmlXPathContextPtr d) {
        ::xmlXPathFreeContext(d);
    };
};

/// Умный указатель вокруг xmlXPathContextPtr из библиотеки %libxml.
typedef xmlPtr<xmlXPathContextPtr> XPathContextPtr;

template<>
struct xml_traits<xmlXPathObjectPtr> {
    static void delete_data(xmlXPathObjectPtr d) {
        xmlXPathFreeObject(d);
    };
};
/// Умный указатель вокруг xmlXPathObjectPtr из библиотеки %libxml.
typedef xmlPtr<xmlXPathObjectPtr> XPathObjectPtr;

template<>
struct xml_traits<xmlDocPtr> {
    static void delete_data(xmlDocPtr d) {
        xmlFreeDoc(d);
    };
};

template<>
struct xml_traits<xmlNodePtr> {
    static void delete_data(xmlNodePtr n) {
        xmlFreeNode(n);
    };
};

template<>
struct xml_traits<xmlURIPtr> {
    static void delete_data(xmlURIPtr u) {
        xmlFreeURI(u);
    }
};

template<>
struct xml_traits<xmlParserCtxtPtr> {
    static void delete_data(xmlParserCtxtPtr n) {
        xmlFreeParserCtxt(n);
    }
};

template<>
struct xml_traits<xmlNodeSetPtr> {
    static void delete_data(xmlNodeSetPtr n) {
        xmlXPathFreeNodeSet(n);
    }
};

/// Умный указатель вокруг xmlChar из библиотеки %libxml.
typedef xmlPtr<xmlChar*> CharPtr;

/// Умный указатель вокруг xmlDocPtr из библиотеки %libxml.
typedef xmlPtr<xmlDocPtr> DocPtr;

/// Умный указатель вокруг xmlURIPtr из библиотеки %libxml.
typedef xmlPtr<xmlURIPtr> URIPtr;

/// Умный указатель вокруг xmlNodePtr из библиотеки %libxml.
typedef xmlPtr<xmlNodePtr> NodePtr;

/// Умный указатель вокруг char*.
typedef xmlPtr<char*> SignedCharPtr;

/// Умный указатель вокруг xmlOutputBufferPtr из библиотеки %libxml.
typedef xmlPtr<xmlOutputBufferPtr> OutputBufferPtr;

/// Умный указатель вокруг xmlParserCtxtPtr из библиотеки %libxml.
typedef xmlPtr<xmlParserCtxtPtr> ParserCtxtPtr;

/// Умный указатель вокруг xmlNodeSetPtr из библиотеки %libxml.
typedef xmlPtr<xmlNodeSetPtr> NodeSetPtr;
};

#endif
