#ifndef RFC2822ADDRESS_PARSER_H
#define RFC2822ADDRESS_PARSER_H

// rfc2822 address parser
// virtan <virtan@virtan.com>

// bnf from rfc2822 section 3.4




template<class STRING, class UV>
class rfc2822address_parser {

    // definition of size_type
public:
    typedef typename STRING::size_type size_type;

    // variables
public:
    const STRING &source;
    const size_type beg;
    const size_type end;
    static const size_type npos;

    // constructor / destructor
public:
    rfc2822address_parser(const STRING &_source, const size_type &_beg, const size_type &_end);
    virtual ~rfc2822address_parser();

    // service functions
public:
    virtual size_type eoi(size_type base, UV uv);
    virtual size_type compare_symbol(size_type base, unsigned char symbol, UV uv);
    virtual size_type compare_range(size_type base, unsigned char since, unsigned char till, UV uv);
    virtual size_type compare_string(size_type base, const char *str, UV uv);
    virtual size_type find_string(size_type base, const char *str, UV uv);

    // parser functions
public:
    virtual size_type address_list(size_type base, UV uv);
    virtual size_type mailbox_list(size_type base, UV uv);
    virtual size_type address(size_type base, UV uv);
    virtual size_type mailbox(size_type base, UV uv);
    virtual size_type simple_addr_spec(size_type base, UV uv);
    virtual size_type name_addr(size_type base, UV uv);
    virtual size_type angle_addr(size_type base, UV uv);
    virtual size_type group(size_type base, UV uv);
    virtual size_type display_name(size_type base, UV uv);
    virtual size_type addr_spec(size_type base, UV uv);
    virtual size_type local_part(size_type base, UV uv);
    virtual size_type domain(size_type base, UV uv);
    virtual size_type domain_literal(size_type base, UV uv);
    virtual size_type dcontent(size_type base, UV uv);
    virtual size_type dtext(size_type base, UV uv);
    virtual size_type dog(size_type base, UV uv);
    virtual size_type dot_atom(size_type base, UV uv);
    virtual size_type dot_atom_text(size_type base, UV uv);
    virtual size_type dot(size_type base, UV uv);
    virtual size_type atext(size_type base, UV uv);
    virtual size_type not_atext(size_type base, UV uv);
    virtual size_type atom(size_type base, UV uv);
    virtual size_type patom(size_type base, UV uv);
    virtual size_type patext(size_type base, UV uv);
    virtual size_type specials(size_type base, UV uv);
    virtual size_type quoted_string(size_type base, UV uv);
    virtual size_type qcontent(size_type base, UV uv);
    virtual size_type qtext(size_type base, UV uv);
    virtual size_type dquote(size_type base, UV uv);
    virtual size_type quoted_pair(size_type base, UV uv);
    virtual size_type phrase(size_type base, UV uv);
    virtual size_type pword(size_type base, UV uv);
    virtual size_type passed_fws(size_type base, UV uv);
    virtual size_type fws(size_type base, UV uv);
    virtual size_type wsp(size_type base, UV uv);
    virtual size_type crlf(size_type base, UV uv);
    virtual size_type cfws(size_type base, UV uv);
    virtual size_type comment(size_type base, UV uv);
    virtual size_type ccontent(size_type base, UV uv);
    virtual size_type cquoted_pair(size_type base, UV uv);
    virtual size_type ctext(size_type base, UV uv);
    virtual size_type obs_addr_list(size_type base, UV uv);
    virtual size_type obs_mbox_list(size_type base, UV uv);
    virtual size_type obs_angle_addr(size_type base, UV uv);
    virtual size_type obs_route(size_type base, UV uv);
    virtual size_type obs_domain_list(size_type base, UV uv);
    virtual size_type obs_domain(size_type base, UV uv);
    virtual size_type html_mailbox(size_type base, UV uv);

    // parser auxiliary functions
private:
    inline size_type address_list_aux_0(size_type base, UV uv);
    inline size_type address_list_aux_1(size_type base, UV uv);
    inline size_type address_list_aux_2(size_type base, UV uv);
    inline size_type address_list_aux_3(size_type base, UV uv);
    inline size_type mailbox_list_aux_0(size_type base, UV uv);
    inline size_type mailbox_list_aux_1(size_type base, UV uv);
    inline size_type mailbox_list_aux_2(size_type base, UV uv);
    inline size_type mailbox_list_aux_3(size_type base, UV uv);
    inline size_type address_aux_0(size_type base, UV uv);
    inline size_type address_aux_1(size_type base, UV uv);
    inline size_type address_aux_2(size_type base, UV uv);
    inline size_type name_addr_aux_0(size_type base, UV uv);
    inline size_type angle_addr_aux_0(size_type base, UV uv);
    inline size_type angle_addr_aux_1(size_type base, UV uv);
    inline size_type group_aux_0(size_type base, UV uv);
    inline size_type group_aux_1(size_type base, UV uv);
    inline size_type domain_literal_aux_0(size_type base, UV uv);
    inline size_type domain_literal_aux_1(size_type base, UV uv);
    inline size_type domain_literal_aux_2(size_type base, UV uv);
    inline size_type domain_literal_aux_3(size_type base, UV uv);
    inline size_type domain_literal_aux_4(size_type base, UV uv);
    inline size_type domain_literal_aux_5(size_type base, UV uv);
    inline size_type dtext_aux_0(size_type base, UV uv);
    inline size_type dot_atom_aux_0(size_type base, UV uv);
    inline size_type dot_atom_aux_1(size_type base, UV uv);
    inline size_type dot_atom_text_aux_0(size_type base, UV uv);
    inline size_type dot_atom_text_aux_1(size_type base, UV uv);
    inline size_type dot_atom_text_aux_2(size_type base, UV uv);
    inline size_type dot_atom_text_aux_3(size_type base, UV uv);
    inline size_type dot_atom_text_aux_4(size_type base, UV uv);
    inline size_type dot_atom_text_aux_5(size_type base, UV uv);
    inline size_type dot_atom_text_aux_6(size_type base, UV uv);
    inline size_type atext_aux_0(size_type base, UV uv);
    inline size_type atom_aux_0(size_type base, UV uv);
    inline size_type atom_aux_1(size_type base, UV uv);
    inline size_type atom_aux_2(size_type base, UV uv);
    inline size_type atom_aux_3(size_type base, UV uv);
    inline size_type patom_aux_0(size_type base, UV uv);
    inline size_type patom_aux_1(size_type base, UV uv);
    inline size_type patom_aux_2(size_type base, UV uv);
    inline size_type patom_aux_3(size_type base, UV uv);
    inline size_type patext_aux_0(size_type base, UV uv);
    inline size_type quoted_string_aux_0(size_type base, UV uv);
    inline size_type quoted_string_aux_1(size_type base, UV uv);
    inline size_type quoted_string_aux_2(size_type base, UV uv);
    inline size_type quoted_string_aux_3(size_type base, UV uv);
    inline size_type quoted_string_aux_4(size_type base, UV uv);
    inline size_type quoted_string_aux_5(size_type base, UV uv);
    inline size_type qtext_aux_0(size_type base, UV uv);
    inline size_type phrase_aux_0(size_type base, UV uv);
    inline size_type phrase_aux_1(size_type base, UV uv);
    inline size_type fws_aux_0(size_type base, UV uv);
    inline size_type fws_aux_1(size_type base, UV uv);
    inline size_type wsp_aux_0(size_type base, UV uv);
    inline size_type wsp_aux_1(size_type base, UV uv);
    inline size_type cfws_aux_0(size_type base, UV uv);
    inline size_type cfws_aux_1(size_type base, UV uv);
    inline size_type cfws_aux_2(size_type base, UV uv);
    inline size_type cfws_aux_3(size_type base, UV uv);
    inline size_type cfws_aux_4(size_type base, UV uv);
    inline size_type comment_aux_0(size_type base, UV uv);
    inline size_type comment_aux_1(size_type base, UV uv);
    inline size_type comment_aux_2(size_type base, UV uv);
    inline size_type comment_aux_3(size_type base, UV uv);
    inline size_type ctext_aux_0(size_type base, UV uv);
    inline size_type obs_addr_list_aux_0(size_type base, UV uv);
    inline size_type obs_addr_list_aux_1(size_type base, UV uv);
    inline size_type obs_addr_list_aux_2(size_type base, UV uv);
    inline size_type obs_addr_list_aux_3(size_type base, UV uv);
    inline size_type obs_addr_list_aux_4(size_type base, UV uv);
    inline size_type obs_mbox_list_aux_0(size_type base, UV uv);
    inline size_type obs_mbox_list_aux_1(size_type base, UV uv);
    inline size_type obs_mbox_list_aux_2(size_type base, UV uv);
    inline size_type obs_mbox_list_aux_3(size_type base, UV uv);
    inline size_type obs_mbox_list_aux_4(size_type base, UV uv);
    inline size_type obs_angle_addr_aux_0(size_type base, UV uv);
    inline size_type obs_angle_addr_aux_1(size_type base, UV uv);
    inline size_type obs_angle_addr_aux_2(size_type base, UV uv);
    inline size_type obs_route_aux_0(size_type base, UV uv);
    inline size_type obs_route_aux_1(size_type base, UV uv);
    inline size_type obs_domain_list_aux_0(size_type base, UV uv);
    inline size_type obs_domain_list_aux_1(size_type base, UV uv);
    inline size_type obs_domain_list_aux_2(size_type base, UV uv);
    inline size_type obs_domain_list_aux_3(size_type base, UV uv);
    inline size_type obs_domain_list_aux_4(size_type base, UV uv);
    inline size_type obs_domain_aux_0(size_type base, UV uv);
    inline size_type obs_domain_aux_1(size_type base, UV uv);
    inline size_type html_mailbox_aux_0(size_type base, UV uv);
    inline size_type html_mailbox_aux_1(size_type base, UV uv);
    inline size_type html_mailbox_aux_2(size_type base, UV uv);

    // not defined functions
public:
    virtual size_type _group_fixer_(size_type base, UV uv)=0;

};


#endif


// constants

template<class STRING, class UV>
const typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::npos(STRING::npos);


// constructor / destructor

template<class STRING, class UV>
rfc2822address_parser<STRING,UV>::rfc2822address_parser(const STRING &_source, const size_type &_beg, const size_type &_end)
    : source(_source), beg(_beg), end(_end)
{
}

template<class STRING, class UV>
rfc2822address_parser<STRING,UV>::~rfc2822address_parser()
{
}


// service functions

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::eoi(size_type base, ATTRIBUTE_UNUSED UV uv)
{
    return base<end?npos:0;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::compare_symbol(size_type base, unsigned char symbol, ATTRIBUTE_UNUSED UV uv)
{
    size_type res;
    if (base>=end) {
        res=npos;
    } else if (symbol) {
        res=source[base]==symbol?1:npos;
    } else {
        res=1;
    }
    return res;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::compare_range(size_type base, unsigned char since, unsigned char till, ATTRIBUTE_UNUSED UV uv)
{
    size_type res;
    if (base>=end) {
        res=npos;
    } else if (source[base]>=since && source[base]<=till) {
        res=1;
    } else {
        res=npos;
    }
    return res;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::compare_string(size_type base, const char *str, ATTRIBUTE_UNUSED UV uv)
{
    int str_len=strlen(str);
    int cmp=source.compare(base,str_len,str);
    return cmp?npos:str_len;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::find_string(size_type base, const char *str, ATTRIBUTE_UNUSED UV uv)
{
    size_type pos=source.find(str,base);
    return pos==npos?pos:pos-base;
}


// parser functions

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::address_list(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=address(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=address_list_aux_2(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=obs_addr_list(offset,uv);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::mailbox_list(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=mailbox(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=mailbox_list_aux_2(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=obs_mbox_list(offset,uv);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::address(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=address_aux_0(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=address_aux_1(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=address_aux_2(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::mailbox(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=name_addr(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=html_mailbox(offset,uv);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=simple_addr_spec(offset,uv);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::simple_addr_spec(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=addr_spec(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::name_addr(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=name_addr_aux_0(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=angle_addr(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::angle_addr(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=angle_addr_aux_0(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=compare_symbol(offset,'<',uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=addr_spec(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=compare_symbol(offset,'>',uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=angle_addr_aux_1(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=obs_angle_addr(offset,uv);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::group(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=display_name(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=compare_symbol(offset,':',uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=_group_fixer_(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=group_aux_0(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=compare_symbol(offset,';',uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=group_aux_1(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::display_name(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=phrase(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::addr_spec(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=local_part(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=dog(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=domain(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::local_part(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=dot_atom(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=quoted_string(offset,uv);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::domain(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=dot_atom(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=domain_literal(offset,uv);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=obs_domain(offset,uv);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::domain_literal(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=domain_literal_aux_0(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=compare_symbol(offset,'[',uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=domain_literal_aux_2(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=domain_literal_aux_4(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=compare_symbol(offset,']',uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=domain_literal_aux_5(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::dcontent(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=dtext(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=quoted_pair(offset,uv);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::dtext(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=dtext_aux_0(offset,uv);
        if (test!=npos) {
            match=false;
        } else if (eoi(offset,uv)==npos) {
            offset+=1;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::dog(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=compare_symbol(offset,'@',uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::dot_atom(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=dot_atom_aux_0(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=dot_atom_text(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=dot_atom_aux_1(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::dot_atom_text(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=dot_atom_text_aux_0(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=dot_atom_text_aux_4(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=dot_atom_text_aux_6(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::dot(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=compare_symbol(offset,'.',uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::atext(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=atext_aux_0(offset,uv);
        if (test!=npos) {
            match=false;
        } else if (eoi(offset,uv)==npos) {
            offset+=1;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::not_atext(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=specials(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=compare_symbol(offset,' ',uv);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::atom(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=atom_aux_0(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=atom_aux_1(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=atom_aux_3(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::patom(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=patom_aux_0(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=patom_aux_1(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=patom_aux_3(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::patext(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=patext_aux_0(offset,uv);
        if (test!=npos) {
            match=false;
        } else if (eoi(offset,uv)==npos) {
            offset+=1;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::specials(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=npos;
        if (test==npos) {
            test=compare_symbol(offset,'(',uv);
        }
        if (test==npos) {
            test=compare_symbol(offset,')',uv);
        }
        if (test==npos) {
            test=compare_symbol(offset,'<',uv);
        }
        if (test==npos) {
            test=compare_symbol(offset,'\076',uv);
        }
        if (test==npos) {
            test=compare_symbol(offset,'[',uv);
        }
        if (test==npos) {
            test=compare_symbol(offset,']',uv);
        }
        if (test==npos) {
            test=compare_symbol(offset,':',uv);
        }
        if (test==npos) {
            test=compare_symbol(offset,'@',uv);
        }
        if (test==npos) {
            test=compare_symbol(offset,';',uv);
        }
        if (test==npos) {
            test=compare_symbol(offset,'\\',uv);
        }
        if (test==npos) {
            test=compare_symbol(offset,',',uv);
        }
        if (test==npos) {
            test=compare_symbol(offset,'.',uv);
        }
        if (test==npos) {
            test=compare_symbol(offset,'\"',uv);
        }
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::quoted_string(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=quoted_string_aux_0(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=dquote(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=quoted_string_aux_2(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=quoted_string_aux_4(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=dquote(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=quoted_string_aux_5(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::qcontent(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=qtext(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=quoted_pair(offset,uv);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::qtext(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=qtext_aux_0(offset,uv);
        if (test!=npos) {
            match=false;
        } else if (eoi(offset,uv)==npos) {
            offset+=1;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::dquote(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=compare_symbol(offset,'\"',uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::quoted_pair(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=compare_symbol(offset,'\\',uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=compare_symbol(offset,0,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::phrase(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=phrase_aux_0(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::pword(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=patom(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=quoted_string(offset,uv);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::passed_fws(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=fws(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::fws(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=fws_aux_0(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::wsp(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=wsp_aux_0(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::crlf(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=compare_symbol(offset,'\r',uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=compare_symbol(offset,'\n',uv);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::cfws(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=cfws_aux_3(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::comment(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=compare_symbol(offset,'(',uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=comment_aux_1(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=comment_aux_3(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=compare_symbol(offset,')',uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::ccontent(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=ctext(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=cquoted_pair(offset,uv);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=comment(offset,uv);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::cquoted_pair(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=compare_symbol(offset,'\\',uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=compare_symbol(offset,0,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::ctext(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=ctext_aux_0(offset,uv);
        if (test!=npos) {
            match=false;
        } else if (eoi(offset,uv)==npos) {
            offset+=1;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::obs_addr_list(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=obs_addr_list_aux_4(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::obs_mbox_list(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=obs_mbox_list_aux_4(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::obs_angle_addr(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=obs_angle_addr_aux_0(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=compare_symbol(offset,'<',uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=obs_angle_addr_aux_1(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=addr_spec(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=compare_symbol(offset,'>',uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=obs_angle_addr_aux_2(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::obs_route(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=obs_route_aux_0(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=obs_domain_list(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=compare_symbol(offset,':',uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=obs_route_aux_1(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::obs_domain_list(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=compare_symbol(offset,'@',uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=domain(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=obs_domain_list_aux_3(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::obs_domain(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=atom(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=obs_domain_aux_0(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::html_mailbox(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=html_mailbox_aux_0(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=compare_symbol(offset,'[',uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=html_mailbox_aux_1(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=compare_string(offset,"mailto:",uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=addr_spec(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=html_mailbox_aux_2(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=compare_symbol(offset,']',uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}


// parser auxiliary functions

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::address_list_aux_0(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=compare_symbol(offset,',',uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=compare_symbol(offset,';',uv);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::address_list_aux_1(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=address(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=cfws(offset,uv);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::address_list_aux_2(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    while (1) {
        test=address_list_aux_3(offset,uv);
        if (test!=npos) {
            offset+=test;
            continue;
        }
        break;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::address_list_aux_3(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=address_list_aux_0(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=address_list_aux_1(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::mailbox_list_aux_0(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=compare_symbol(offset,',',uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=compare_symbol(offset,';',uv);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::mailbox_list_aux_1(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=mailbox(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=cfws(offset,uv);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::mailbox_list_aux_2(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    while (1) {
        test=mailbox_list_aux_3(offset,uv);
        if (test!=npos) {
            offset+=test;
            continue;
        }
        break;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::mailbox_list_aux_3(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=mailbox_list_aux_0(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=mailbox_list_aux_1(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::address_aux_0(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=cfws(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::address_aux_1(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=mailbox(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=group(offset,uv);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::address_aux_2(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=cfws(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::name_addr_aux_0(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=display_name(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::angle_addr_aux_0(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=cfws(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::angle_addr_aux_1(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=cfws(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::group_aux_0(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=mailbox_list(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=cfws(offset,uv);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::group_aux_1(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=cfws(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::domain_literal_aux_0(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=cfws(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::domain_literal_aux_1(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=fws(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::domain_literal_aux_2(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    while (1) {
        test=domain_literal_aux_3(offset,uv);
        if (test!=npos) {
            offset+=test;
            continue;
        }
        break;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::domain_literal_aux_3(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=domain_literal_aux_1(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=dcontent(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::domain_literal_aux_4(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=fws(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::domain_literal_aux_5(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=cfws(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::dtext_aux_0(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=npos;
        if (test==npos) {
            test=compare_symbol(offset,' ',uv);
        }
        if (test==npos) {
            test=compare_symbol(offset,'\t',uv);
        }
        if (test==npos) {
            test=compare_symbol(offset,'\r',uv);
        }
        if (test==npos) {
            test=compare_symbol(offset,'\n',uv);
        }
        if (test==npos) {
            test=compare_symbol(offset,'[',uv);
        }
        if (test==npos) {
            test=compare_symbol(offset,']',uv);
        }
        if (test==npos) {
            test=compare_symbol(offset,'\\',uv);
        }
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::dot_atom_aux_0(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=cfws(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::dot_atom_aux_1(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=cfws(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::dot_atom_text_aux_0(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    int cycle=0;
    while (++cycle) {
        test=dot_atom_text_aux_1(offset,uv);
        if (test!=npos) {
            offset+=test;
            continue;
        }
        break;
    }
    if (cycle==1) {
        match=false;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::dot_atom_text_aux_1(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=atext(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::dot_atom_text_aux_2(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    int cycle=0;
    while (++cycle) {
        test=dot_atom_text_aux_3(offset,uv);
        if (test!=npos) {
            offset+=test;
            continue;
        }
        break;
    }
    if (cycle==1) {
        match=false;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::dot_atom_text_aux_3(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=atext(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::dot_atom_text_aux_4(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    while (1) {
        test=dot_atom_text_aux_5(offset,uv);
        if (test!=npos) {
            offset+=test;
            continue;
        }
        break;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::dot_atom_text_aux_5(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=dot(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=dot_atom_text_aux_2(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::dot_atom_text_aux_6(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=dot(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::atext_aux_0(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=not_atext(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::atom_aux_0(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=cfws(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::atom_aux_1(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    int cycle=0;
    while (++cycle) {
        test=atom_aux_2(offset,uv);
        if (test!=npos) {
            offset+=test;
            continue;
        }
        break;
    }
    if (cycle==1) {
        match=false;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::atom_aux_2(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=atext(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::atom_aux_3(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=cfws(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::patom_aux_0(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=cfws(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::patom_aux_1(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    int cycle=0;
    while (++cycle) {
        test=patom_aux_2(offset,uv);
        if (test!=npos) {
            offset+=test;
            continue;
        }
        break;
    }
    if (cycle==1) {
        match=false;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::patom_aux_2(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=patext(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::patom_aux_3(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=cfws(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::patext_aux_0(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=npos;
        if (test==npos) {
            test=compare_symbol(offset,'(',uv);
        }
        if (test==npos) {
            test=compare_symbol(offset,')',uv);
        }
        if (test==npos) {
            test=compare_symbol(offset,'<',uv);
        }
        if (test==npos) {
            test=compare_symbol(offset,'\076',uv);
        }
        if (test==npos) {
            test=compare_symbol(offset,'[',uv);
        }
        if (test==npos) {
            test=compare_symbol(offset,']',uv);
        }
        if (test==npos) {
            test=compare_symbol(offset,':',uv);
        }
        if (test==npos) {
            test=compare_symbol(offset,';',uv);
        }
        if (test==npos) {
            test=compare_symbol(offset,'\\',uv);
        }
        if (test==npos) {
            test=compare_symbol(offset,',',uv);
        }
        if (test==npos) {
            test=compare_symbol(offset,'\"',uv);
        }
        if (test==npos) {
            test=compare_symbol(offset,' ',uv);
        }
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::quoted_string_aux_0(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=cfws(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::quoted_string_aux_1(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=passed_fws(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::quoted_string_aux_2(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    while (1) {
        test=quoted_string_aux_3(offset,uv);
        if (test!=npos) {
            offset+=test;
            continue;
        }
        break;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::quoted_string_aux_3(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=quoted_string_aux_1(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=qcontent(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::quoted_string_aux_4(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=passed_fws(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::quoted_string_aux_5(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=cfws(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::qtext_aux_0(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=npos;
        if (test==npos) {
            test=compare_symbol(offset,' ',uv);
        }
        if (test==npos) {
            test=compare_symbol(offset,'\t',uv);
        }
        if (test==npos) {
            test=compare_symbol(offset,'\r',uv);
        }
        if (test==npos) {
            test=compare_symbol(offset,'\n',uv);
        }
        if (test==npos) {
            test=compare_symbol(offset,'\\',uv);
        }
        if (test==npos) {
            test=compare_symbol(offset,'\"',uv);
        }
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::phrase_aux_0(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    int cycle=0;
    while (++cycle) {
        test=phrase_aux_1(offset,uv);
        if (test!=npos) {
            offset+=test;
            continue;
        }
        break;
    }
    if (cycle==1) {
        match=false;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::phrase_aux_1(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=pword(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::fws_aux_0(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    int cycle=0;
    while (++cycle) {
        test=fws_aux_1(offset,uv);
        if (test!=npos) {
            offset+=test;
            continue;
        }
        break;
    }
    if (cycle==1) {
        match=false;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::fws_aux_1(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=wsp(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=crlf(offset,uv);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::wsp_aux_0(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    int cycle=0;
    while (++cycle) {
        test=wsp_aux_1(offset,uv);
        if (test!=npos) {
            offset+=test;
            continue;
        }
        break;
    }
    if (cycle==1) {
        match=false;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::wsp_aux_1(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=npos;
        if (test==npos) {
            test=compare_symbol(offset,' ',uv);
        }
        if (test==npos) {
            test=compare_symbol(offset,'\t',uv);
        }
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::cfws_aux_0(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=fws(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::cfws_aux_1(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    int cycle=0;
    while (++cycle) {
        test=cfws_aux_2(offset,uv);
        if (test!=npos) {
            offset+=test;
            continue;
        }
        break;
    }
    if (cycle==1) {
        match=false;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::cfws_aux_2(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=cfws_aux_0(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=comment(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::cfws_aux_3(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    int cycle=0;
    while (++cycle) {
        test=cfws_aux_4(offset,uv);
        if (test!=npos) {
            offset+=test;
            continue;
        }
        break;
    }
    if (cycle==1) {
        match=false;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::cfws_aux_4(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=cfws_aux_1(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=fws(offset,uv);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::comment_aux_0(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=fws(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::comment_aux_1(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    while (1) {
        test=comment_aux_2(offset,uv);
        if (test!=npos) {
            offset+=test;
            continue;
        }
        break;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::comment_aux_2(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=comment_aux_0(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=ccontent(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::comment_aux_3(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=fws(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::ctext_aux_0(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=npos;
        if (test==npos) {
            test=compare_symbol(offset,'(',uv);
        }
        if (test==npos) {
            test=compare_symbol(offset,')',uv);
        }
        if (test==npos) {
            test=compare_symbol(offset,'\\',uv);
        }
        if (test==npos) {
            test=compare_symbol(offset,' ',uv);
        }
        if (test==npos) {
            test=compare_symbol(offset,'\r',uv);
        }
        if (test==npos) {
            test=compare_symbol(offset,'\n',uv);
        }
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::obs_addr_list_aux_0(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=cfws(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::obs_addr_list_aux_1(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=cfws(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::obs_addr_list_aux_2(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    while (1) {
        test=obs_addr_list_aux_3(offset,uv);
        if (test!=npos) {
            offset+=test;
            continue;
        }
        break;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::obs_addr_list_aux_3(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=obs_addr_list_aux_0(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=compare_symbol(offset,',',uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=obs_addr_list_aux_1(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=address(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::obs_addr_list_aux_4(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=address(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=obs_addr_list_aux_2(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::obs_mbox_list_aux_0(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=cfws(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::obs_mbox_list_aux_1(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=cfws(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::obs_mbox_list_aux_2(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    while (1) {
        test=obs_mbox_list_aux_3(offset,uv);
        if (test!=npos) {
            offset+=test;
            continue;
        }
        break;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::obs_mbox_list_aux_3(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=obs_mbox_list_aux_0(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=compare_symbol(offset,',',uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=obs_mbox_list_aux_1(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=mailbox(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::obs_mbox_list_aux_4(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=mailbox(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=obs_mbox_list_aux_2(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::obs_angle_addr_aux_0(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=cfws(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::obs_angle_addr_aux_1(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=obs_route(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::obs_angle_addr_aux_2(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=cfws(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::obs_route_aux_0(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=cfws(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::obs_route_aux_1(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=cfws(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::obs_domain_list_aux_0(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    while (1) {
        test=obs_domain_list_aux_1(offset,uv);
        if (test!=npos) {
            offset+=test;
            continue;
        }
        break;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::obs_domain_list_aux_1(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=cfws(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=dot(offset,uv);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::obs_domain_list_aux_2(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=cfws(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::obs_domain_list_aux_3(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    while (1) {
        test=obs_domain_list_aux_4(offset,uv);
        if (test!=npos) {
            offset+=test;
            continue;
        }
        break;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::obs_domain_list_aux_4(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=obs_domain_list_aux_0(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=obs_domain_list_aux_2(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=compare_symbol(offset,'@',uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=domain(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::obs_domain_aux_0(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    while (1) {
        test=obs_domain_aux_1(offset,uv);
        if (test!=npos) {
            offset+=test;
            continue;
        }
        break;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::obs_domain_aux_1(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=dot(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=atom(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::html_mailbox_aux_0(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=display_name(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::html_mailbox_aux_1(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=cfws(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING, class UV>
typename rfc2822address_parser<STRING,UV>::size_type rfc2822address_parser<STRING,UV>::html_mailbox_aux_2(size_type base, UV uv)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=cfws(offset,uv);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

