#ifndef RFC2822DATE_PARSER_H
#define RFC2822DATE_PARSER_H


/*

$Log$
Revision 1.1  2006/10/24 13:50:38  virtan
rfc2822* moved to mimeparser

Revision 1.2  2005/03/31 13:02:49  virtan
rfc2822 address parsing added

Revision 1.1  2005/03/29 12:42:22  virtan
Now we rely on rfc2822 date syntax


*/


// rfc2822 date parser
// virtan <virtan@virtan.com>

// bnf from rfc2822 section 3.3


namespace rfc2822 {

template<class STRING>
class rfc2822date_parser {

    // definition of size_type
public:
    typedef typename STRING::size_type size_type;

    // variables
public:
    const STRING &source;
    const size_type beg;
    const size_type end;
    static const size_type npos;

    // constructor / destructor
public:
    rfc2822date_parser(const STRING &_source, const size_type &_beg, const size_type &_end);
    virtual ~rfc2822date_parser();

    // service functions
public:
    virtual size_type eoi(size_type base);
    virtual size_type compare_symbol(size_type base, char symbol);
    virtual size_type compare_range(size_type base, char since, char till);
    virtual size_type compare_string(size_type base, const char *str);
    virtual size_type find_string(size_type base, const char *str);

    // parser functions
public:
    virtual size_type date_time(size_type base);
    virtual size_type day_of_week(size_type base);
    virtual size_type day_name(size_type base);
    virtual size_type date(size_type base);
    virtual size_type day(size_type base);
    virtual size_type month(size_type base);
    virtual size_type month_name(size_type base);
    virtual size_type year(size_type base);
    virtual size_type time(size_type base);
    virtual size_type zone(size_type base);
    virtual size_type time_of_day(size_type base);
    virtual size_type hour(size_type base);
    virtual size_type minute(size_type base);
    virtual size_type second(size_type base);
    virtual size_type short_date_time(size_type base);
    virtual size_type month_digits(size_type base);
    virtual size_type obs_day_of_week(size_type base);
    virtual size_type obs_day(size_type base);
    virtual size_type obs_month(size_type base);
    virtual size_type obs_year(size_type base);
    virtual size_type obs_hour(size_type base);
    virtual size_type obs_minute(size_type base);
    virtual size_type obs_second(size_type base);
    virtual size_type obs_zone(size_type base);
    virtual size_type day_digits(size_type base);
    virtual size_type year_digits(size_type base);
    virtual size_type old_year_digits(size_type base);
    virtual size_type hour_digits(size_type base);
    virtual size_type minute_digits(size_type base);
    virtual size_type second_digits(size_type base);
    virtual size_type digit(size_type base);
    virtual size_type digit2(size_type base);
    virtual size_type alpha(size_type base);
    virtual size_type fws(size_type base);
    virtual size_type obs_fws(size_type base);
    virtual size_type wsp(size_type base);
    virtual size_type crlf(size_type base);
    virtual size_type cfws(size_type base);
    virtual size_type comment(size_type base);
    virtual size_type ccontent(size_type base);
    virtual size_type ctext(size_type base);
    virtual size_type quoted_pair(size_type base);
    virtual size_type jan(size_type base);
    virtual size_type feb(size_type base);
    virtual size_type mar(size_type base);
    virtual size_type apr(size_type base);
    virtual size_type may(size_type base);
    virtual size_type jun(size_type base);
    virtual size_type jul(size_type base);
    virtual size_type aug(size_type base);
    virtual size_type sep(size_type base);
    virtual size_type oct(size_type base);
    virtual size_type nov(size_type base);
    virtual size_type dec(size_type base);
    virtual size_type zone_digits(size_type base);
    virtual size_type old_zone(size_type base);
    virtual size_type gmt(size_type base);
    virtual size_type est(size_type base);
    virtual size_type edt(size_type base);
    virtual size_type cst(size_type base);
    virtual size_type cdt(size_type base);
    virtual size_type mst(size_type base);
    virtual size_type mdt(size_type base);
    virtual size_type pst(size_type base);
    virtual size_type pdt(size_type base);
    virtual size_type zone_military(size_type base);

    // parser auxiliary functions
private:
    inline size_type date_time_aux_0(size_type base);
    inline size_type date_time_aux_1(size_type base);
    inline size_type date_time_aux_2(size_type base);
    inline size_type day_of_week_aux_0(size_type base);
    inline size_type day_of_week_aux_1(size_type base);
    inline size_type day_aux_0(size_type base);
    inline size_type day_aux_1(size_type base);
    inline size_type month_aux_0(size_type base);
    inline size_type time_aux_0(size_type base);
    inline size_type time_of_day_aux_0(size_type base);
    inline size_type short_date_time_aux_0(size_type base);
    inline size_type short_date_time_aux_1(size_type base);
    inline size_type short_date_time_aux_2(size_type base);
    inline size_type short_date_time_aux_3(size_type base);
    inline size_type short_date_time_aux_4(size_type base);
    inline size_type short_date_time_aux_5(size_type base);
    inline size_type month_digits_aux_0(size_type base);
    inline size_type obs_day_of_week_aux_0(size_type base);
    inline size_type obs_day_of_week_aux_1(size_type base);
    inline size_type obs_day_aux_0(size_type base);
    inline size_type obs_year_aux_0(size_type base);
    inline size_type obs_hour_aux_0(size_type base);
    inline size_type obs_hour_aux_1(size_type base);
    inline size_type obs_minute_aux_0(size_type base);
    inline size_type obs_minute_aux_1(size_type base);
    inline size_type obs_second_aux_0(size_type base);
    inline size_type day_digits_aux_0(size_type base);
    inline size_type fws_aux_0(size_type base);
    inline size_type fws_aux_1(size_type base);
    inline size_type obs_fws_aux_0(size_type base);
    inline size_type obs_fws_aux_1(size_type base);
    inline size_type wsp_aux_0(size_type base);
    inline size_type crlf_aux_0(size_type base);
    inline size_type cfws_aux_0(size_type base);
    inline size_type cfws_aux_1(size_type base);
    inline size_type comment_aux_0(size_type base);
    inline size_type comment_aux_1(size_type base);
    inline size_type comment_aux_2(size_type base);
    inline size_type ctext_aux_0(size_type base);
    inline size_type zone_digits_aux_0(size_type base);

};





// constants

template<class STRING>
const typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::npos(STRING::npos);


// constructor / destructor

template<class STRING>
rfc2822date_parser<STRING>::rfc2822date_parser(const STRING &_source, const size_type &_beg, const size_type &_end)
    : source(_source), beg(_beg), end(_end)
{
}

template<class STRING>
rfc2822date_parser<STRING>::~rfc2822date_parser()
{
}


// service functions

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::eoi(size_type base)
{
    return base<end?npos:0;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::compare_symbol(size_type base, char symbol)
{
    if (base>=end) {
        return npos;
    }
    if (symbol) {
        return source[base]==symbol?1:npos;
    }
    return 1;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::compare_range(size_type base, char since, char till)
{
    if (base>=end) {
        return npos;
    }
    if (source[base]>=since && source[base]<=till) {
        return 1;
    }
    return npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::compare_string(size_type base, const char *str)
{
    int str_len=strlen(str);
    int cmp=source.compare(base,str_len,str);
    return cmp?npos:str_len;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::find_string(size_type base, const char *str)
{
    size_type pos=source.find(str,base);
    return pos==npos?pos:pos-base;
}


// parser functions

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::date_time(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=date_time_aux_1(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=date(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=cfws(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=time(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=date_time_aux_2(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=short_date_time(offset);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::day_of_week(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=day_of_week_aux_1(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=obs_day_of_week(offset);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::day_name(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=alpha(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=alpha(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=alpha(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::date(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=day(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=month(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=year(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::day(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=day_aux_1(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=obs_day(offset);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::month(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=month_aux_0(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=obs_month(offset);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::month_name(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=jan(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=feb(offset);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=mar(offset);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=apr(offset);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=may(offset);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=jun(offset);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=jul(offset);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=aug(offset);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=sep(offset);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=oct(offset);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=nov(offset);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=dec(offset);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::year(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=year_digits(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=obs_year(offset);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::time(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=time_of_day(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=time_aux_0(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::zone(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=zone_digits(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=obs_zone(offset);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::time_of_day(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=hour(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=compare_symbol(offset,':');
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=minute(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=time_of_day_aux_0(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::hour(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=hour_digits(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=obs_hour(offset);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::minute(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=minute_digits(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=obs_minute(offset);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::second(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=second_digits(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=obs_second(offset);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::short_date_time(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=obs_day(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=short_date_time_aux_0(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=compare_symbol(offset,'.');
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=short_date_time_aux_1(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=month_digits(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=short_date_time_aux_2(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=compare_symbol(offset,'.');
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=short_date_time_aux_3(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=year(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=cfws(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=obs_hour(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=compare_symbol(offset,':');
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=obs_minute(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=short_date_time_aux_4(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=short_date_time_aux_5(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::month_digits(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=digit(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=month_digits_aux_0(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::obs_day_of_week(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=obs_day_of_week_aux_0(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=day_name(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=obs_day_of_week_aux_1(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::obs_day(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=obs_day_aux_0(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=day_digits(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::obs_month(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=cfws(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=month_name(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=cfws(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::obs_year(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=obs_year_aux_0(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=old_year_digits(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::obs_hour(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=obs_hour_aux_0(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=hour_digits(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=obs_hour_aux_1(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::obs_minute(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=obs_minute_aux_0(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=minute_digits(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=obs_minute_aux_1(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::obs_second(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=obs_second_aux_0(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=second_digits(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::obs_zone(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=old_zone(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::day_digits(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=digit(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=day_digits_aux_0(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::year_digits(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=digit2(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=digit2(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::old_year_digits(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=digit2(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::hour_digits(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=digit2(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::minute_digits(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=digit2(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::second_digits(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=digit2(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::digit(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=npos;
        if (test==npos) {
            test=compare_range(offset,'0','9');
        }
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::digit2(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=digit(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=digit(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::alpha(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=npos;
        if (test==npos) {
            test=compare_range(offset,'A','Z');
        }
        if (test==npos) {
            test=compare_range(offset,'a','z');
        }
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::fws(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=fws_aux_1(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=wsp(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=obs_fws(offset);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::obs_fws(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=wsp(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=obs_fws_aux_1(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::wsp(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=wsp_aux_0(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::crlf(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=crlf_aux_0(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=compare_symbol(offset,'\n');
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::cfws(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=cfws_aux_1(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=fws(offset);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::comment(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=compare_symbol(offset,'(');
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=comment_aux_1(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=comment_aux_2(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=compare_symbol(offset,')');
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::ccontent(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=ctext(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=quoted_pair(offset);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=comment(offset);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::ctext(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=ctext_aux_0(offset);
        if (test!=npos) {
            match=false;
        } else {
            offset+=1;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::quoted_pair(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=compare_symbol(offset,'\\');
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=compare_symbol(offset,0);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::jan(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=compare_string(offset,"Jan");
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::feb(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=compare_string(offset,"Feb");
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::mar(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=compare_string(offset,"Mar");
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::apr(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=compare_string(offset,"Apr");
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::may(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=compare_string(offset,"May");
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::jun(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=compare_string(offset,"Jun");
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::jul(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=compare_string(offset,"Jul");
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::aug(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=compare_string(offset,"Aug");
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::sep(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=compare_string(offset,"Sep");
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::oct(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=compare_string(offset,"Oct");
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::nov(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=compare_string(offset,"Nov");
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::dec(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=compare_string(offset,"Dec");
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::zone_digits(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=zone_digits_aux_0(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=digit2(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=digit2(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::old_zone(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=gmt(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=est(offset);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=edt(offset);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=cst(offset);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=cdt(offset);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=mst(offset);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=mdt(offset);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=pst(offset);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=pdt(offset);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=zone_military(offset);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::gmt(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=compare_string(offset,"UT");
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=compare_string(offset,"GMT");
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::est(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=compare_string(offset,"EST");
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::edt(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=compare_string(offset,"EDT");
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::cst(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=compare_string(offset,"CST");
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::cdt(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=compare_string(offset,"CDT");
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::mst(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=compare_string(offset,"MST");
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::mdt(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=compare_string(offset,"MDT");
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::pst(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=compare_string(offset,"PST");
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::pdt(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=compare_string(offset,"PDT");
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::zone_military(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=alpha(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}


// parser auxiliary functions

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::date_time_aux_0(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=compare_symbol(offset,',');
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::date_time_aux_1(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=day_of_week(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=date_time_aux_0(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::date_time_aux_2(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=cfws(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::day_of_week_aux_0(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=fws(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::day_of_week_aux_1(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=day_of_week_aux_0(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=day_name(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::day_aux_0(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=fws(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::day_aux_1(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=day_aux_0(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=day_digits(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::month_aux_0(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=fws(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=month_name(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=fws(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::time_aux_0(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=cfws(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=zone(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::time_of_day_aux_0(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=compare_symbol(offset,':');
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=second(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::short_date_time_aux_0(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=cfws(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::short_date_time_aux_1(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=cfws(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::short_date_time_aux_2(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=cfws(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::short_date_time_aux_3(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=cfws(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::short_date_time_aux_4(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=compare_symbol(offset,':');
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=obs_second(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::short_date_time_aux_5(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=cfws(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::month_digits_aux_0(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=digit(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::obs_day_of_week_aux_0(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=cfws(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::obs_day_of_week_aux_1(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=cfws(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::obs_day_aux_0(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=cfws(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::obs_year_aux_0(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=cfws(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::obs_hour_aux_0(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=cfws(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::obs_hour_aux_1(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=cfws(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::obs_minute_aux_0(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=cfws(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::obs_minute_aux_1(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=cfws(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::obs_second_aux_0(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=cfws(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::day_digits_aux_0(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=digit(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::fws_aux_0(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    while (1) {
        if (match) {
            test=wsp(offset);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
        if (match) {
            continue;
        }
        break;
    }
    match=true;
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::fws_aux_1(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=fws_aux_0(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (match) {
        test=crlf(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::obs_fws_aux_0(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    int cycle=0;
    while (++cycle) {
        if (match) {
            test=wsp(offset);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
        if (match) {
            continue;
        }
        break;
    }
    if (cycle!=1) {
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::obs_fws_aux_1(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    while (1) {
        if (match) {
            test=crlf(offset);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
        if (match) {
            test=obs_fws_aux_0(offset);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
        if (match) {
            continue;
        }
        break;
    }
    match=true;
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::wsp_aux_0(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    int cycle=0;
    while (++cycle) {
        if (match) {
            test=npos;
            if (test==npos) {
                test=compare_symbol(offset,' ');
            }
            if (test==npos) {
                test=compare_symbol(offset,'\t');
            }
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
        if (match) {
            continue;
        }
        break;
    }
    if (cycle!=1) {
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::crlf_aux_0(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=compare_symbol(offset,'\r');
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::cfws_aux_0(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=fws(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::cfws_aux_1(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    int cycle=0;
    while (++cycle) {
        if (match) {
            test=cfws_aux_0(offset);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
        if (match) {
            test=comment(offset);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
        if (match) {
            continue;
        }
        break;
    }
    if (cycle!=1) {
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::comment_aux_0(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=fws(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::comment_aux_1(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    while (1) {
        if (match) {
            test=comment_aux_0(offset);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
        if (match) {
            test=ccontent(offset);
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
        if (match) {
            continue;
        }
        break;
    }
    match=true;
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::comment_aux_2(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=fws(offset);
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::ctext_aux_0(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=npos;
        if (test==npos) {
            test=compare_symbol(offset,'(');
        }
        if (test==npos) {
            test=compare_symbol(offset,')');
        }
        if (test==npos) {
            test=compare_symbol(offset,'\\');
        }
        if (test==npos) {
            test=compare_symbol(offset,' ');
        }
        if (test==npos) {
            test=compare_symbol(offset,'\r');
        }
        if (test==npos) {
            test=compare_symbol(offset,'\n');
        }
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    return match?offset-base:npos;
}

template<class STRING>
typename rfc2822date_parser<STRING>::size_type rfc2822date_parser<STRING>::zone_digits_aux_0(size_type base)
{
    size_type offset=base;
    size_type test;
    bool match=true;
    if (match) {
        test=compare_symbol(offset,'+');
        if (test!=npos) {
            offset+=test;
        } else {
            match=false;
        }
    }
    if (!match) {
        offset=base;
        match=true;
        if (match) {
            test=compare_symbol(offset,'-');
            if (test!=npos) {
                offset+=test;
            } else {
                match=false;
            }
        }
    }
    return match?offset-base:npos;
}

}

#endif
