/*******************************************************************************
 *  The Elm Mail System  -  $Revision$   $State$
 *
 *                      Copyright (c) 1988-1995 USENET Community Trust
 *          Copyright (c) 1986,1987 Dave Taylor
 *******************************************************************************
 * Bug reports, patches, comments, suggestions should be sent to:
 *
 *      Bill Pemberton, Elm Coordinator
 *      flash@virginia.edu
 *
 *******************************************************************************
 * $Log$
 * Revision 1.3  2007/07/03 09:02:37  virtan
 * rfc2822 parse bug fix
 *
 * Revision 1.2  2007/04/23 12:14:25  virtan
 * avoiding using mimetic
 *
 * Revision 1.4  1996/05/09  15:51:09  wfp5p
 * Alpha 10
 *
 * Revision 1.3  1995/09/29  17:41:25  wfp5p
 * Alpha 8 (Chip's big changes)
 *
 * Revision 1.2  1995/09/11  15:18:56  wfp5p
 * Alpha 7
 *
 * Revision 1.1  1995/07/18  18:59:50  wfp5p
 * Alpha 6
 *
 *
 *
 ******************************************************************************/


/*
 * parse_arpa_mailbox() parses RFC-822 "mailbox" specifications into address
 * and fullname components.  A "mailbox" is the formal name of the RFC-822
 * lexical element that corresponds to what we normally might call an
 * "address".  (RFC-822 uses the term "address" to describe something else.)
 *
 * A "mailbox" can be in one of two formats:
 *
 *   addr-spec
 *       such as:  joe@acme.com (Joe User)
 *
 * or:
 *
 *   [phrase] "<" [route] addr-spec ">"
 *       such as:  Joe User <joe@acme.com>
 *
 * We invent the names "bare addr-spec" to describe the first form and
 * "angle addr-spec" to describe the second.
 *
 * Synopsis:
 *
 *   int parse_arpa_mailbox(buf, ret_addr, len_addr,
 *      ret_name, len_name, next_field);
 *   char *buf, *ret_addr, *ret_name, **next_field;
 *   int len_addr, len_name;
 *
 * This routine takes a comma-delimited list of mailbox specifications
 * pointed to by "buf", and breaks the next mailbox specification in the
 * list into the address and fullname components.  It is NONdestructive
 * to the buffer.
 *
 * The return code will be 0 for success, -1 for failure.  All bets are
 * off if the mailbox specification is poorly formed (i.e. syntax errors).
 * We might catch the problem and return -1.  Or we might indicate success
 * and return nonsense values.  Other error conditions are discussed
 * below.
 *
 * If "ret_addr" is not NULL, then it points to a buffer where the
 * extracted address is stored and the "len_addr" indicates the size of
 * the buffer.  If we cannot locate a non-empty address or if it is too
 * large to fit into the buffer, then an error is returned.
 *
 * If "ret_name" is not NULL, then it points to a buffer where the
 * extracted fullname is stored and the "len_name" indicates the size of
 * the buffer.  If we cannot locate a non-empty fullname or if it is too
 * large to fit into the buffer, then an empty string is stored in the
 * buffer.  We need to discard the fullname rather than truncating it
 * because truncation could result in an illegal value (e.g. unbalanced
 * quotes).  As an added little glitch, if the fullname value is fully
 * enclosed in double-quotes (and with no interior double-quotes), then
 * the quotes will be stripped.
 *
 * If the "next_field" pointer is not NULL, it will be set to point to
 * the beginning of the next mailbox specification in the list.  It will
 * point to the '\0' string terminator when the list is complete.  This
 * update occurs even if an error code is returned, thus address parsing
 * may continue with the next mailbox in the list.  If the "buf" contains
 * a single address, the "next_field" result should be checked upon return
 * to ensure it points to the '\0' string terminator.
 */

#include <mimeparser/rfc2822address.h>
#include <cstring>
#include <algorithm>

#include <boost/range/algorithm/copy.hpp>
#include <boost/range/adaptor/transformed.hpp>
#include <boost/algorithm/string/join.hpp>

#include <recipient_parser/parse.hpp>

namespace rfc2822 {

int parse_bare_addrspec(const char *, std::string &, std::string &, char **);
int parse_angle_addrspec(const char *, std::string &, std::string &, char **);
int rfc822_toklen(const char *str);
int parse_arpa_mailbox(const char *buf, std::string &ret_addr, std::string &ret_name, char **next_field);

static bool parse_group(const char *buffer, std::string &name, address_list_t &result, char **next)
{
    const char *i = buffer;
    for (; isspace(*i); ++i) {
        ;
    }
    const char *name_begin = i, *name_end = 0;
    for (std::string::size_type l = 0; *i != 0 && *i != ',' && *i != ':' && *i != ';'; l = rfc822_toklen(i)) {
        if (isspace(*i)) {
            i += l;
        } else {
            name_end = (i += l);
        }
    }
    if (*i != ':') {
        return false;
    }
    if (name_begin && name_end && name.empty()) {
        name.assign(name_begin, name_end);
    }
    char *end;
    for (i++; *i != 0 && *i != ';'; i = end) {
        std::string name, address;
        if (parse_arpa_mailbox(i, address, name, &end) >= 0) {
            result.push_back(std::make_pair(name, address));
        }
    }
    *next = const_cast<char *>(i) + 1;
    return true;
}

void old_rfc2822address::parse(const std::string &buffer)
{
    char *next;
    for (const char *begin = buffer.c_str(); begin < buffer.c_str() + buffer.size() && *begin != 0; begin = next) {
        if (parse_group(begin, gr_name_, *this, &next)) {
            continue;
        }
        std::string name, address;
        if (parse_arpa_mailbox(begin, address, name, &next) >= 0) {
            push_back(std::make_pair(name, address));
        }
    }
    status_ = true;
}

#define MAX_LINE_LENGTH 1024




/*
 * This is like strncpy() except the result is guaranteed to be '\0' terminated.
 */
char *strfcpy(char *dest,const char * src,int len)
{
    (void) strncpy(dest, src, len);
    dest[len-1] = '\0';
    return dest;
}

int parse_arpa_mailbox(const char *buf, std::string &ret_addr, std::string &ret_name, char **next_field)
{
    const char *s;
    int rc;
    /*
     * Take a quick look through the buffer to determine the format.
     */
    for (s = buf ; *s != '\0' && *s != '<' && *s != ',' ; s += rfc822_toklen(s)) {
        ;
    }
    /*
     * Handle as the appropriate format.
     */
    if (*s == '<') {
        rc = parse_angle_addrspec(buf, ret_addr, ret_name, next_field);
    } else {
        rc = parse_bare_addrspec(buf, ret_addr, ret_name, next_field);
    }
    /*
     * In the case of an error, advance to next mailbox field.
     */
    if (rc < 0 && next_field != 0) {
        for (s = buf ; *s != '\0' && *s != ',' ; s += rfc822_toklen(s)) {
            ;
        }
        if (*s == ',') {
            ++s;
        }
        *next_field = const_cast<char *>(s);
    }
    /*
      dprint(5, (debugfile, "parse_arpa_mailbox - addr=\"%s\" name=\"%s\"\n",
      (ret_addr != 0 ? ret_addr : "(null)"),
      (ret_name != 0 ? ret_name : "(null)")));
    */
    return rc;
}

inline bool atom(char c)
{
    return !((c > 0 && c <= ' ') || c == '(' || c == '"' || c == ')' || c == ',' || c == '.' || c == ':' || c == ';' || c == '<' || c == '>' || c == '[' || c == ']' || c == '@');
}

std::string quote(const std::string &source)
{
    bool quoted = false;
    std::string result, space;
    result.reserve(source.length() + 2);
    std::string::size_type i = 0;
    for (; i < source.length() && source[i] == ' '; result += source[i++]) {
        ;
    }
    result += '\"';
    for (; i < source.length(); ++i)
        if (source[i] == ' ') {
            space += source[i];
        } else {
            result += space;
            space.clear();
            if (source[i] == '\\' || source[i] == '\"') {
                result += '\\';
                quoted = true;
            } else if (!atom(source[i]) && source[i] != '.') {
                quoted = true;
            }
            result += source[i];
        }
    result += '\"';
    result += space;
    if (quoted) {
        return result;
    }
    return source;
}

bool unquote_pair(const std::string &source, std::string::size_type &pos, std::string &result)
{
    if (pos + 1 > source.length() || source[pos] != '\\') {
        return false;
    }
    pos++;
    result += source[pos++];
    return true;
}

bool quoted_pair(const std::string &source, std::string::size_type &pos, std::string &result)
{
    if (pos + 1 > source.length() || source[pos] != '\\') {
        return false;
    }
    result += source[pos++];
    result += source[pos++];
    return true;
}

bool spaces(const std::string &source, std::string::size_type &pos, std::string &result)
{
    std::string::size_type tmp_pos = pos;
    while (tmp_pos < source.length() && (source[tmp_pos] == ' ' || source[tmp_pos] == '\t' || source[tmp_pos] == '\r' || source[tmp_pos] == '\n')) {
        tmp_pos++;
    }
    if (tmp_pos == pos) {
        return false;
    }
    pos = tmp_pos;
    if (*result.rbegin() != ' ') {
        result += ' ';
    }
    return true;
}

bool unquote(const std::string &source, std::string::size_type &pos, std::string &result)
{
    if (pos >= source.length() || source[pos] != '\"') {
        return false;
    }
    std::string::size_type tmp_pos = pos + 1;
    std::string tmp_result;
    while (tmp_pos < source.length() && source[tmp_pos] != '\"')
        if (!unquote_pair(source, tmp_pos, tmp_result) && !spaces(source, tmp_pos, tmp_result)) {
            tmp_result += source[tmp_pos++];
        }
    if (source[tmp_pos] != '\"') {
        return false;
    }
    result += tmp_result;
    pos = tmp_pos + 1;
    return true;
}

std::string unquote(const std::string& str)
{
    std::string result;
    size_t pos = 0;
    if (unquote(str, pos, result)) {
        return result;
    } else {
        return str;
    }
}

bool comment(const std::string &source, std::string::size_type &pos, std::string &result)
{
    if (pos >= source.length() || source[pos] != '(') {
        return false;
    }
    std::string::size_type tmp_pos = pos;
    std::string tmp_result;
    int level = 0;
    tmp_result += source[tmp_pos++];
    while (tmp_pos < source.length() && source[tmp_pos] != ')') {
        if (source[tmp_pos] == '(') {
            level++;
        } else if (source[tmp_pos] == ')') {
            level--;
        }
        if (!quoted_pair(source, tmp_pos, tmp_result) && !spaces(source, tmp_pos, tmp_result)) {
            tmp_result += source[tmp_pos++];
        }
    }
    if (source[tmp_pos] != ')' || level > 0) {
        return false;
    }
    tmp_result += source[tmp_pos++];
    result += tmp_result;
    pos = tmp_pos;
    return true;
}

bool atom(const std::string &source, std::string::size_type &pos, std::string &result)
{
    std::string::size_type tmp_pos = pos;
    std::string tmp_result;
    while (tmp_pos < source.length() && (atom(source[tmp_pos]) || source[tmp_pos] == '.')) {
        tmp_result += source[tmp_pos++];
    }
    if (tmp_pos == pos) {
        return false;
    }
    result += tmp_result;
    pos = tmp_pos;
    return true;
}

void parse_name(const std::string &source, std::string &name)
{
    std::string::size_type pos = 0;
    std::string data;
    while (pos < source.size()) {
        std::string space, comment_data, tmp_data;
        spaces(source, pos, space);
        if (!atom(source, pos, tmp_data) && !unquote(source, pos, tmp_data)) {
            if (comment(source, pos, comment_data)) {
                spaces(source, pos, comment_data);
                data += space;
                data += comment_data;
            } else if (source[pos] == '\"' || source[pos] == '(') {
                data += space + source[pos++];
            } else {
                break;
            }
        } else if (tmp_data.empty()) {
            data += space;
        } else {
            data += space + tmp_data;
        }
    }
    if (!data.empty()) {
        name = quote(data);
    }
}

bool domain_atom(const std::string &source, std::string::size_type &pos)
{
    std::string::size_type tmp_pos = pos;
    while (tmp_pos < source.length() && source[tmp_pos] != '\'' && (atom(source[tmp_pos]) || source[tmp_pos] == '.')) {
        tmp_pos++;
    }
    if (tmp_pos == pos || (tmp_pos < source.length() && source[tmp_pos] == '\'')) {
        return false;
    }
    pos = tmp_pos;
    return true;
}

bool domain_literal(const std::string &source, std::string::size_type &pos)
{
    if (pos >= source.length() || source[pos] != '[') {
        return false;
    }
    std::string::size_type tmp_pos = pos;
    std::string tmp_result;
    while (tmp_pos < source.length() && source[tmp_pos] != ']') {
        if (source[tmp_pos] == '[') {
            return false;
        }
        if (!quoted_pair(source, tmp_pos, tmp_result) && !spaces(source, tmp_pos, tmp_result)) {
            tmp_pos++;
        }
    }
    if (source[tmp_pos] != ']') {
        return false;
    }
    pos = tmp_pos;
    return true;
}

bool parse_address(const std::string &source)
{
    std::string::size_type pos = 0;
    std::string local, space;
    if (!atom(source, pos, local) && !unquote(source, pos, local)) {
        return false;
    }
    spaces(source, pos, space);
    if (pos < source.length() && source[pos] != '@') {
        return false;
    }
    spaces(source, ++pos, space);
    if (!domain_atom(source, pos) && !domain_literal(source, pos)) {
        return false;
    }
    return true;
}

/*
 * Parse a mailbox spec in the format:  addr-spec
 */
int parse_bare_addrspec(const char *buf, std::string &ret_addr, std::string &ret_name, char **next_field)
{
    const char *n_ptr = buf;    /* pointer to (user name) into "buf"    */
    const char *n_ptr_end = n_ptr;
    int tlen;       /* length of current token      */
    int got_addr;       /* indicates an address was found   */
    /*
     * We will set "n_ptr" to the right-most occurance of (parens)
     * encountered when scanning the buffer.  We go back later and
     * extract this into the "ret_name" buffer.
     */
    /*
     * We will set this TRUE when we discover there really is an addr here.
     */
    got_addr = 0;
    /*
     * Discard leading space.
     */
    while (*buf && isspace(*buf)) {
        ++buf;
    }
    /*
     * Scan through the field, copying out the address elements.
     */
    while (*buf != '\0' && *buf != ',' && *buf!=';') {
        tlen = rfc822_toklen(buf);
        if (isspace(*buf)) {
            /*
             * Discard whitespace.
             */
            ; /* nop */
        } else if (*buf == '(') { /*)*/
            /*
             * Save info so we can go back later and extract
             * the right-most comment with (parens) stripped.
             */
            n_ptr = buf+1;
            n_ptr_end = std::find(n_ptr, n_ptr + strlen(n_ptr), ')');
        } else {
            ret_addr.append(buf, tlen);
            got_addr = 1;
        }
        buf += tlen;
    }
    /*
     * Make sure we extracted a valid address and terminate the string.
     */
    if (!got_addr) {
        return -1;
    }
    /*
     * If there is a fullname comment then save it off, else set the result
     * to an empty string.  Also return an empty string if the buffer isn't
     * big enough.  That's because if we only stored off a portion, we
     * could end up with something like unbalanced quotes.
     */
    if (n_ptr != n_ptr_end) {
        std::string source(n_ptr, n_ptr_end);
        parse_name(source, ret_name);
    }
    /*
     * We should be at the end of the mailbox field.
     */
    if (*buf != '\0' && *buf != ',' && *buf!=';') {
        return -1;
    }
    /*
     * Save off pointer to next mailbox field.
     */
    if (next_field != 0) {
        *next_field = const_cast<char *>(buf) + (*buf == ',' ? 1 : 0);
    }
    return 0;
}

/*
 * Parse a mailbox spec in the format:  [phrase] "<" [route] addr-spec ">"
 */
int parse_angle_addrspec(const char *buf, std::string &ret_addr, std::string &ret_name, char **next_field)
{
    const char *beg_field, *end_field;
    int tlen;
    int w;
    /*
     * Discard leading space.
     */
    while (isspace(*buf)) {
        ++buf;
    }
    /*
     * Locate the front and back of the fullname portion.
     * "end_field" actually points one beyond the end of the field.
     */
    beg_field = end_field = buf;
    while (*buf != '<' && *buf != '\0' && *buf != ',' && *buf!=';') {
        tlen = rfc822_toklen(buf);
        /*
         * By updating "end_field" only on non-space tokens we ensure
         * that when we copy beg_field->end_field trailing whitespace
         * will be elided.
         */
        if (!isspace(*buf)) {
            buf += tlen;
            end_field = buf;
        } else {
            buf += tlen;
        }
    }
    if (*buf != '<') {
        return -1;
    }
    /*
     * If there is a fullname field then save it off, else set the result
     * to an empty string.  Also return an empty string if the buffer isn't
     * big enough.  That's because if we only stored off a portion, we
     * could end up with something like unbalanced quotes.
     */
    const char *n_beg = NULL, *n_end = NULL;
    if ((w = static_cast<int>(end_field - beg_field)) > 0 && w < MAX_LINE_LENGTH) {
        parse_name(std::string(beg_field, w), ret_name);
        n_beg = beg_field;
        n_end = end_field;
    }
    /*
     * Locate the front and back of the address field.
     * "end_field" actually points one beyond the end of the field.
     */
    beg_field = ++buf;
    while (*buf != '>' && *buf != '\0') {
        buf += rfc822_toklen(buf);
    }
    if (*buf != '>') {
        return -1;
    }
    end_field = buf;
    /*
     * Calculate the length of the address and save off the result.
     */
    if ((w = static_cast<int>(end_field - beg_field)) < 0) {
        return -1;
    }
    {
        ret_addr.assign(beg_field, w);
        if (!parse_address(ret_addr) && n_beg && n_end && parse_address(std::string(n_beg, n_end))) {
            parse_name(ret_addr, ret_name);
            ret_addr.assign(n_beg, n_end - n_beg);
        } else {
            std::string::size_type pos1 = ret_addr.find_first_not_of(" \t"), pos2 = ret_addr.find_last_not_of(" \t");
            if (pos1 != std::string::npos && pos2 != std::string::npos && pos2 - pos1 > 1)
                if ((ret_addr[pos1] == '\"' && ret_addr[pos2] == '\"') || (ret_addr[pos1] == '\'' && ret_addr[pos2] == '\'')) {
                    ret_addr = ret_addr.substr(pos1 + 1, pos2 - pos1 - 1);
                }
        }
    }
    for (++buf; isspace(*buf); ++buf) {
        ;
    }
    beg_field = buf;
    for (; *buf != '\0' && *buf != ',' && *buf != ';'; buf += rfc822_toklen(buf)) {
        ;
    }
    if (ret_name.empty() && buf > beg_field) {
        parse_name(std::string(beg_field, buf), ret_name);
    }
    /*
     * Save off pointer to next mailbox field.
     */
    if (next_field != 0) {
        *next_field = const_cast<char *>(buf) + (*buf == ',' ? 1 : 0);
    }
    return 0;
}




/*******************************************************************************
 *  The Elm Mail System  -  $Revision$   $State$
 *
 *                      Copyright (c) 1988-1995 USENET Community Trust
 *******************************************************************************
 * Bug reports, patches, comments, suggestions should be sent to:
 *
 *      Bill Pemberton, Elm Coordinator
 *      flash@virginia.edu
 *
 *******************************************************************************
 * $Log$
 * Revision 1.3  2007/07/03 09:02:37  virtan
 * rfc2822 parse bug fix
 *
 * Revision 1.2  2007/04/23 12:14:25  virtan
 * avoiding using mimetic
 *
 * Revision 1.3  1995/09/29  17:41:35  wfp5p
 * Alpha 8 (Chip's big changes)
 *
 * Revision 1.2  1995/09/11  15:18:58  wfp5p
 * Alpha 7
 *
 * Revision 1.1.1.1  1995/04/19  20:38:33  wfp5p
 * Initial import of elm 2.4 PL0 as base for elm 2.5.
 *
 ******************************************************************************/


/*
 * rfc822_toklen(str) - Returns length of RFC-822 token that starts at "str".
 *
 * We understand the following tokens:
 *
 *  linear-white-space
 *  "quoted string"
 *  [dom.ain.lit.eral]
 *  (comment)
 *  \c (quoted character)
 *  control characters
 *  special characters (other chars with semantic meaning in addresses)
 *  atom (strings of alphanumerics and non-special/non-control chars)
 *
 * This routine is a profiling hot spot.  To speed things up, a lookup
 * table is used to classify the character types.  The table is initialized
 * the first time this routine is called.
 *
 * At this time, this routine does not do any error handling, and will
 * process defective tokens (e.g. no closing paren or quote).  Grep
 * for ERROR to see the places where error handling should be added if
 * it ever is necessary.
 */

#define charlen(s)  ((s)[0] == '\\' && (s)[1] != '\0' ? 2 : 1)

#define CH_EOS      0   /* \0 - we should not see this!     */
#define CH_ATOM     1   /* char that can be part of an atom */
#define CH_SPACE    2   /* linear white space character     */
#define CH_COMMENT  3   /* ( char - comment         */
#define CH_QSTR     4   /* " char - quoted string       */
#define CH_QCHAR    5   /* \ char - quoted character        */
#define CH_DOMLIT   6   /* [ char - domain literal      */
#define CH_SPECIAL  7   /* some other char with special meaning */
#define CH_CTL      8   /* a non-printing control character */

#define chtype(c)   (chtab[(c) & 0377])

/*
 * Assuming headers only contain 7-bit US-ASCII, which
 * should be true for the structured address fields.
 */
char chtab[0400];

class init_chtab {
public:
    init_chtab(void) {
        int i = 0;
        /* most chars in the range 001 - 037 are control chars */
        while (i < 040) {
            chtab[i++] = CH_CTL;
        }
        /* most char in the range 040 - 0377 are "atom" chars */
        while (i < 0400) {
            chtab[i++] = CH_ATOM;
        }
        chtab[0] = CH_EOS;
        /* mark whitespace chars */
        chtab[static_cast<unsigned int>(' ')] = CH_SPACE;
        chtab[static_cast<unsigned int>('\t')] = CH_SPACE;
        chtab[static_cast<unsigned int>('\r')] = CH_SPACE;
        chtab[static_cast<unsigned int>('\n')] = CH_SPACE;
        /* mark special chars that require further lexical processing */
        chtab[static_cast<unsigned int>('"')] = CH_QSTR;
        chtab[static_cast<unsigned int>('(')] = CH_COMMENT;
        chtab[static_cast<unsigned int>('[')] = CH_DOMLIT;
        chtab[static_cast<unsigned int>('\\')] = CH_QCHAR;
        /* mark remaining chars that are special in address fields */
        chtab[static_cast<unsigned int>(')')] = CH_SPECIAL;
        chtab[static_cast<unsigned int>('<')] = CH_SPECIAL;
        chtab[static_cast<unsigned int>('>')] = CH_SPECIAL;
        chtab[static_cast<unsigned int>('@')] = CH_SPECIAL;
        chtab[static_cast<unsigned int>(',')] = CH_SPECIAL;
        chtab[static_cast<unsigned int>(';')] = CH_SPECIAL;
        chtab[static_cast<unsigned int>(':')] = CH_SPECIAL;
        chtab[static_cast<unsigned int>('.')] = CH_SPECIAL;
        chtab[static_cast<unsigned int>(']')] = CH_SPECIAL;
    }
};

init_chtab x;



int rfc822_toklen(const char *str)
{
    const char *str0;
    int depth;
    str0 = str;
    switch (chtype(*str)) {
        case CH_ATOM:
            do {
                ++str;
            } while (chtype(*str) == CH_ATOM);
            return static_cast<int>(str-str0);
        case CH_SPACE:
            do {
                ++str;
            } while (chtype(*str) == CH_SPACE);
            return static_cast<int>(str-str0);
        case CH_COMMENT:
            ++str;
            depth = 0;
            while (*str != '\0' && (*str != ')' || depth > 0)) {
                switch (*str) {
                    case '(':
                        ++str;
                        ++depth;
                        break;
                    case ')':
                        ++str;
                        --depth;
                        break;
                    default:
                        str += charlen(str);
                        break;
                }
            }
            if (*str == ')') {
                ++str;
            } else {
                ;    /* ERROR - unterminated paren */
            }
            return static_cast<int>(str-str0);
        case CH_QSTR:
            ++str;
            while (*str != '\0' && *str != '"') {
                str += charlen(str);
            }
            if (*str == '"') {
                ++str;
            } else {
                ;    /* ERROR - unterminated quote */
            }
            return static_cast<int>(str-str0);
        case CH_QCHAR:
            if (str[1] != '\0') {
                return 2;
            }
            return 1; /* ERROR - string ends with backslash */
        case CH_DOMLIT:
            ++str;
            while (*str != '\0' && *str != ']') {
                str += charlen(str);
            }
            if (*str == ']') {
                ++str;
            } else {
                ;    /* ERROR - unterminated domain literal */
            }
            return static_cast<int>(str-str0);
        case CH_EOS:
            if (*str != '\0') {
                return 1;    /* 0x80 and not really end of string */
            }
            return 0; /* ERROR - we should not see this */
            /* case CH_SPECIAL: */
            /* case CH_CTL: */
        default:
            return 1;
    }
    /*NOTREACHED*/
}

void rfc2822address::parse(const std::string& src) {
    auto iter = src.begin();
    const auto end = src.end();

    auto result = rcpt_parser::parse_address_list(iter, end);
    if( !result || iter != end ) {
#ifdef MIMEPARSER_DEBUG_PRINT_FALLBACK
        std::cerr << "new_address_iterator parse failed, src=[" << src << "], "
                "parser stopped here=[" << std::string(iter, end) << "]" << std::endl;
#endif /*MIMEPARSER_DEBUG_PRINT_FALLBACK*/
        parse_fallback(src);
    } else {
        status_ = true;
        boost::copy(
            rcpt_parser::flat_mailbox_list(*result)
              | boost::adaptors::transformed([](const rcpt_parser::types::NameAddr& nameAddr) {
                    return address_pair_t{
                        quote(boost::algorithm::join(nameAddr.display_name, " ")),
                        nameAddr.addr_spec.login + "@" + nameAddr.addr_spec.domain
                    };
                }),
            std::back_inserter(*this)
        );
    }
}

void rfc2822address::parse_fallback(const std::string& src) {
    old_rfc2822address other(src);
    status_ = other.ok();
    if( ok() ) {
        swap(other);
    }
}


#ifdef _TEST

#include <stdio.h>
#include <iostream>

int debug = 0;
FILE *debugfile = stderr;
main()
{
    char b;
    string s;
    while (cin.get(b)) {
        s+=b;
    }
    rfc2822address a(s);
    cout<<"Gname : "<<a.group_name()<<endl;
    for (address_list_t::iterator it=a.begin(); it!=a.end(); ++it) {
        cerr<<"fname: "<<it->first<<" addr: "<<it->second<<endl;
    }
}
#endif /*_TEST*/
};

