#include <memory>

#include <boost/python.hpp>

#include <mimeparser/rfc2822address.h>

namespace mimeparser {

using AddressList = rfc2822::rfc2822address;

struct Recipient {
    const std::string name;
    const std::string address;
};

class RecipientIterator {
public:
    RecipientIterator(const std::shared_ptr<AddressList>& addresses)
        : addresses(addresses), current(addresses->begin()) {}

    Recipient next() {
        if (current == addresses->end()) {
            PyErr_SetString(PyExc_StopIteration, "No more recipients.");
            boost::python::throw_error_already_set();
        }
        const Recipient result {current->first, current->second};
        ++current;
        return result;
    }

private:
    std::shared_ptr<AddressList> addresses;
    AddressList::const_iterator current;
};

class ParseRecipients {
public:
    ParseRecipients(const std::string& str)
        : addresses(std::make_shared<AddressList>(str)) {}

    RecipientIterator iter() const {
        return RecipientIterator(addresses);
    }

private:
    std::shared_ptr<AddressList> addresses;
};

ParseRecipients parseRecipients(const std::string& str) {
    return ParseRecipients(str);
}

} // namespace mimeparser

BOOST_PYTHON_MODULE(mimeparser) {
    using namespace boost::python;
    using namespace mimeparser;

    class_<Recipient>("Recipient", no_init)
        .def_readonly("name", &Recipient::name)
        .def_readonly("address", &Recipient::address)
        ;

    class_<RecipientIterator>("RecipientIterator", no_init)
        .def("next", &RecipientIterator::next)
        ;

    class_<ParseRecipients>("ParseRecipients", no_init)
        .def("__iter__", &ParseRecipients::iter)
        ;

    def("parse_recipients", parseRecipients);
}
