#include <mimeparser/HeaderData.h>

#include <mimeparser/StringTools.h>
#include <mimeparser/HeaderField.h>

#include <boost/algorithm/string/predicate.hpp>

using namespace MimeParser;
using rfc2822::rfc2822address;

HeaderData::
HeaderData(bool /*unfolded*/)
    : m_contentType("text")
    , m_contentSubtype("plain")
    , m_charset("US-ASCII")
    , m_content_transfer_encoding("7bit")
{}

const std::string& HeaderData::cType() const
{
    return m_contentType;
}
const std::string& HeaderData::contentSubtype() const
{
    return m_contentSubtype;
}
const std::string& HeaderData::boundary() const
{
    return m_boundary;
}
const std::string& HeaderData::charset() const
{
    return m_charset;
}
const mulca_mime::DecodedString& HeaderData::name() const
{
    return m_name;
}
const std::string& HeaderData::content_transfer_encoding() const
{
    return m_content_transfer_encoding;
}
const std::string& HeaderData::content_disposition() const
{
    return m_content_disposition;
}
const mulca_mime::DecodedString& HeaderData::filename() const
{
    return m_filename;
}
const std::string& HeaderData::content_id() const
{
    return m_content_id;
}
const rfc2822::rfc2822address* HeaderData::from() const
{
    return m_from.get();
}
const rfc2822::rfc2822address* HeaderData::to() const
{
    return m_to.get();
}
const rfc2822::rfc2822address* HeaderData::cc() const
{
    return m_cc.get();
}
const std::string& HeaderData::messageId() const
{
    return m_messageId;
}
const HeaderData::HeaderFields& HeaderData::headerFields() const
{
    return m_headerFields;
}
bool HeaderData::isInlineMessage() const
{
    using boost::iequals;
    return iequals("message", m_contentType) &&
           iequals("rfc822", m_contentSubtype);
}

bool
HeaderData::
parseContentType(const HeaderField& headerField)
{
    const std::string& value=headerField.value();
    const std::string& name=headerField.name();
    if (boost::iequals("content-type", name)) {
        mulca_mime::parameters_ng p(value, 0, value.size(), m_rawHeadersList);
        if (!p.token().empty()) {
            m_contentType=p.token();
        }
        if (!p.token2().empty()) {
            m_contentSubtype=p.token2();
        }
        if (boost::iequals("multipart", m_contentType)) {
            if (p.exist("boundary")) {
                m_boundary=p["boundary"].contents;
            }
        }
        if (p.exist("charset")) {
            m_charset=p["charset"].contents;
        }
        m_name=p["name"];
        return true;
    } else {
        return false;
    }
}

bool
HeaderData::
parseContentDisposition(const HeaderField& headerField)
{
    const std::string& value=headerField.value();
    const std::string& name=headerField.name();
    if (boost::iequals("content-disposition", name)) {
        mulca_mime::parameters_ng p(value, 0, value.size(), m_rawHeadersList);
        m_content_disposition=p.token();
        m_filename=p["filename"];
        return true;
    } else {
        return false;
    }
}

bool
HeaderData::
parseContentTransferEncoding(const HeaderField& headerField)
{
    const std::string& value=headerField.value();
    const std::string& name=headerField.name();
    if (boost::iequals("content-transfer-encoding", name)) {
        m_content_transfer_encoding.assign(trim(value));
        return true;
    } else {
        return false;
    }
}

bool
HeaderData::
parseContentId(const HeaderField& headerField)
{
    const std::string& value=headerField.value();
    const std::string& name=headerField.name();
    if (boost::iequals("content-id", name)) {
        m_content_id.assign(trim(value));
        return true;
    } else {
        return false;
    }
}

bool
HeaderData::
parseFrom(const HeaderField& headerField)
{
    m_from.reset(new rfc2822address(headerField.value()));
    return NULL!=m_from.get();
}

bool
HeaderData::
parseTo(const HeaderField& headerField)
{
    m_to.reset(new rfc2822address(headerField.value()));
    return NULL!=m_to.get();
}

bool
HeaderData::
parseCc(const HeaderField& headerField)
{
    m_cc.reset(new rfc2822address(headerField.value()));
    return NULL!=m_cc.get();
}

bool
HeaderData::
parseMessageId(const HeaderField& headerField)
{
    m_messageId=trim(headerField.value());
    return true;
}

bool
HeaderData::
addHeaderField(const HeaderField& headerField)
{
    m_headerFields.push_back(headerField);
    return true;
}

void
HeaderData::
setRawHeaderList(const HeaderList& headers)
{
    m_rawHeadersList = headers;
}
