//
// -*- C++ -*-
//

#include <butil/email/helpers.h>
#include <mimeparser/build_msg.h>
#include <iconv.h>
#include <errno.h>
#include <vector>


using std::string;


sequence<int> int_seq;
string local_hostname::hname;
local_hostname lh;

bool atomchar(char c)
{
    return !((c > 0  && c <= ' ') || c == '(' || c=='"' || c ==')'
             || c == ',' || c == '.' || c == ':' || c == ';' || c == '<'
             || c == '>' || c == '[' || c == ']' || c == '@');
}

string quote(const string& src)
{
    bool quoted = false;
    string dst, space;
    dst.reserve(src.size() + 2);
    size_t i = 0;
    while (i < src.size() && src[i] == ' ') {
        dst += src[i++];
    }
    dst += '\"';
    for (; i < src.size(); ++i) {
        if (src[i] == ' ') {
            space += src[i];
        } else {
            dst += space;
            space.clear();
            if (src[i] == '\\' || src[i] == '\"') {
                dst += '\\';
                quoted = true;
            } else if (!atomchar(src[i]) && src[i] != '.') {
                quoted = true;
            }
            dst += src[i];
        }
    }
    dst += '\"';
    dst += space;
    if (quoted) {
        return dst;
    } else {
        return src;
    }
}

string delimeter(bool folding)
{
    return folding ? (string(",")+string(crln)+"\t") : ", ";
}

string
transform(const string& value,const string& charset, bool folding)
{
    rfc2822::rfc2822address ap(value);
    string res;
    for (rfc2822::rfc2822address::iterator i=ap.begin(); i!=ap.end();) {
        if (!i->first.empty()) {
            string display = i->first;
            if (!charset.empty() && have_8bit(display)) {
                display = rfc2822::unquote(display);
                display = transform_2047(display, charset, "base64", true);
            }
            res +=display + " <"+i->second+">";
        } else {
            res +=i->second;
        }
        ++i;
        if (i!=ap.end()) {
            res += delimeter(folding);
        } else {
            break;
        }
    }
    return res;
}

string transform_email_header_draft_field(const string& value, const string& charset) {
    string res;
    try {
        rfc2822ns::address_iterator it(value), end;
        while (it != end) {
            const std::string domain = idna::encode(it.domain());
            const std::string login = it.local();
            std::string display = it.display();

            if (!display.empty() && !charset.empty() && have_8bit(display)) {
                display = rfc2822::unquote(display);
                display = transform_2047(display, charset, "base64", false);
            }
            if (!res.empty()) {
                res += delimeter(true);
            }
            if (!login.empty() || !domain.empty()) {
                res += rfc2822ns::join_address(display, login, domain);
            } else {
                res += display;
            }

            try {
                ++it;
            } catch (const rfc2822ns::invalid_address &err) {
                if (!res.empty()) {
                    res += delimeter(true);
                }
                res += transform_2047(err.what(), charset, "base64", false);
                break;
            }
        }
    } catch (...) {
        res = transform_2047(value, charset, "base64", false);
    }
    return res;
}

string transform_email_header_field(const string& value,const string& charset) {
    return transform(value, charset, true);
}

string transform_email_header_field_nofolding(const string& value,const string& charset) {
    return transform(value, charset, false);
}


const string::size_type MAX_LINE=60;

string
transform_2047(const string& name,const string& charset,const string&, bool do_on8)
{
    // XXX Ignoring encoding (only base64 now).
    if (charset.empty() || (do_on8 && !have_8bit(name))) {
        return name;
    }
    string prefix("=?"+charset+"?B?");
    string suffix("?=");
    string::size_type p=0;
    string res;
    bool failed = true;
    iconv_t i = iconv_open(charset.c_str(), charset.c_str());
    if (i != reinterpret_cast<iconv_t>(-1)) {
        failed = false;
        char *inptr = const_cast<char *>(name.data());
        size_t inlen = name.size();
        while (inlen > 0) {
            std::vector<char> out(MAX_LINE, '\0');
            char *outptr = &out[0];
            size_t outlen = out.size();
            size_t rc = iconv(i, &inptr, &inlen, &outptr, &outlen);
            if (rc != size_t(-1) || errno == E2BIG) {
                failed = false;
                if (res.size() > 0) {
                    res += string(crln) + "\t";
                }
                res += prefix
                       + encode_base64(string(out.begin(),
                                              out.begin() + (outptr - &out[0])))
                       + suffix;
            } else {
                failed = true;
                break;
            }
        }
        iconv_close(i);
    }
    if (failed) {
        res = "";
        while (p<name.size()) {
            if (p!=0) {
                res+=string(crln)+"\t";
            }
            res+=prefix+encode_base64(name.substr(p,MAX_LINE))+suffix;
            p+=MAX_LINE;
        }
    }
    return res;
}



