#include <iostream>
#include <assert.h>
#include <mimeparser/part.h>
#include <mimeparser/ccnv.h>
#include <boost/lexical_cast.hpp>


namespace mulca_mime {

using namespace std;
typedef std::string::size_type size_type;

part::~part()
{
}

std::string::size_type
find_boundary(const std::string& buffer,const std::string& bon, const std::string::size_type beg,
              const std::string::size_type end, std::string::size_type& nbeg,
              bool& is_final)
{
    std::string::size_type start=beg;
    std::string::size_type pos=std::string::npos;
    while (start < end) {
        nbeg=pos= buffer.find(bon, start);
        if (pos==std::string::npos || (pos+bon.size() > end)) {
            return std::string::npos;
        }
        pos += bon.size();
        if ((pos+1 < end) && (buffer[pos] == '-') && (buffer[pos+1] == '-')) {
            pos +=2;
            is_final=true;
        } else {
            is_final=false;
        }
        if (nbeg - 1 >= beg && buffer[nbeg - 1] == '\n') {
            --nbeg;
            if (nbeg - 1 >= beg && buffer[nbeg - 1] == '\r') {
                --nbeg;
            }
        }
        while (pos < end && (buffer[pos] == ' ' || buffer[pos] == '\t' || buffer[pos]=='\r')) {
            ++pos;
        }
        if (pos  < end && buffer[pos] =='\n') {
            ++pos;
            return pos;
        } else if (pos == end) { // Boundary at eof
            return pos;
        } else {
            start=pos;
            continue;
        }
    }
    return std::string::npos;
}

struct part::parse_data {
    bool use_inline_level;
    int inline_level;
    int level;
    size_type beg;
    size_type end;
    bool final;
    bool* fake;
};

size_type part::do_parse(const part::parse_data& d)
{
    if (d.use_inline_level) {
        inlineLevel = d.inline_level;
    }
    size_type newEnd = header.parse(inlineLevel, d.beg, d.end);
    offset() = newEnd;
    length() = d.end - offset();
    if (d.level > 20) { // XXX Enough?
        return d.end;
    }
    if (d.final && header.field_count() == 0) {
        // This targets RFC-incompliant messages that lack a closing boundary;
        //   we discard the last part of such messages unless the header
        //   is not empty.
        if (d.fake) {
            *d.fake = true;
        }
        return d.end;
    }
    size_type nbeg;
    size_type nend;
    if (d.use_inline_level &&
            strcasecmp(header.content_type.type.c_str(),"message")==0 &&
            strcasecmp(header.content_type.subtype.c_str(),"rfc822")==0 &&
            inlineLevel > 0) {
        part p(buffer, this);
        parse_data dd = { d.use_inline_level, inlineLevel-1, d.level+1, newEnd,
                          d.end, false, 0
                        };
        newEnd = p.do_parse(dd);
        parts.push_back(p);
        return newEnd;
    } else if ((strcasecmp(header.content_type.type.c_str(),"multipart")!=0)
               || header.content_type.boundary.empty()) {
        return body.parse(inlineLevel, newEnd, d.end);
    } else {
        const std::string b = "--" + header.content_type.boundary;
        bool final=false;
        nbeg = find_boundary(buffer,b,newEnd,d.end,nend,final);
        if (nbeg == std::string::npos) {
            return d.end;
        }
        size_type pos;
        do {
            pos=find_boundary(buffer,b,nbeg,d.end,nend,final);
            if (pos==std::string::npos) { // missing final boundary!
                nend=d.end;
                final=true;
            }
            part p(buffer, this);
            bool fake = false;
            parse_data dd = { d.use_inline_level, inlineLevel, d.level+1, nbeg,
                              nend, final, &fake
                            };
            newEnd = p.do_parse(dd);
            nbeg = pos;
            if (!d.use_inline_level &&
                    parts.size() > max_part) {
                final=true;
            }
            if (!fake) {
                parts.push_back(p);
            }
        } while ((nbeg < d.end) && !final);
    }
    return d.end;
}

size_type part::parse(int ilevel, const size_type beg, const size_type end,
                      int level)
{
    parse_data d =  { true, ilevel, level, beg, end, false, NULL };
    return do_parse(d);
}

size_type part::parse(const size_type beg, const size_type end, int level)
{
    parse_data d =  { false, 0, level, beg, end, false, NULL };
    return do_parse(d);
}


std::ostream& operator<<(std::ostream& o, const part& p)
{
    p.dump(o, "1");
    return o;
}

void part_str(std::string& o, const part& p)
{
    p.dump(o, "1");
}

namespace {
inline void
print_field(const std::string& name, const std::string& value,
            const std::string& tab, std::ostream& os)
{
    if (!value.empty()) {
        os << tab << "\t" << name << "=\""
           << xml10_escape_utf8_string(value) << '"' << endl;
    }
}

inline void
print_field(const std::string& name, const std::string& value,
            const std::string& tab, std::string& o)
{
    if (!value.empty()) {
        o += tab;
        o+="\t";
        o +=name;
        o+="=\"";
        o += xml10_escape_utf8_string(value);
        o += "\"\n";
    }
}
} // namespace

void
part::dump(std::ostream& os, const std::string& id_prefix, unsigned int, unsigned int level) const
{
    std::string tab(level, '\t');
    os << tab << "<part id=\"" << id_prefix << "\" offset=\"" << offset()
       << "\" length=\"" << length() << "\"" << endl;
    print_field("content_type.type",header.content_type.type,tab,os);
    print_field("content_type.subtype",header.content_type.subtype,tab,os);
    print_field("content_type.charset",header.content_type.charset,tab,os);
    print_field("content_type.name",header.content_type.name,tab,os);
    print_field("content_transfer_encoding",header.content_transfer_encoding,tab,os);
    print_field("content_disposition.value",header.content_disposition.value,tab,os);
    print_field("content_disposition.filename",header.content_disposition.filename,
                tab,os);
    print_field("content_id",header.content_id,tab,os);
    os << tab << ">" << endl;
    unsigned j = 1;
    for (parts_t::const_iterator i = parts.begin(); i!= parts.end(); ++j, ++i) {
        i->dump(os, id_prefix + "." + boost::lexical_cast<std::string>(j), j, level+1);
    }
    os << endl << tab << "</part>" << endl;
}

void
part::dump(std::string& o, const std::string& id_prefix, unsigned int, unsigned int level) const
{
    std::string tab(level, '\t');
    o +=tab;
    o +="<part id=\"";
    o +=id_prefix;
    o +="\" offset=\"";
    o +=boost::lexical_cast<std::string>(offset());
    o +="\" length=\"";
    o +=boost::lexical_cast<std::string>(length());
    o +="\"\n" ;
    print_field("content_type.type",header.content_type.type,tab,o);
    print_field("content_type.subtype",header.content_type.subtype,tab,o);
    print_field("content_type.charset",header.content_type.charset,tab,o);
    print_field("content_type.name",header.content_type.name,tab,o);
    print_field("content_transfer_encoding",header.content_transfer_encoding,tab,o);
    print_field("content_disposition.value",header.content_disposition.value,tab,o);
    print_field("content_disposition.filename",header.content_disposition.filename,
                tab,o);
    print_field("content_id",header.content_id,tab,o);
    o +=tab;
    o +=">\n";
    unsigned j = 1;
    for (parts_t::const_iterator i = parts.begin(); i!= parts.end(); ++j, ++i) {
        i->dump(o, id_prefix + "." + boost::lexical_cast<std::string>(j), j, level+1);
    }
    o +="\n";
    o +=tab;
    o +="</part>\n";
}

} // namespace mulca_mime
