#include <iostream>
#include <sys/time.h>
#include <time.h>
#include <mimeparser/rfc2822date.h>
#ifdef ARCADIA_BUILD
#include <util/datetime/systime.h>
#endif

/* Mail parse date into elt fields
 * Accepts: elt to write into
 *          date string to parse
 * Returns: T if parse successful, else NIL
 * This routine parses dates as follows:
 * . leading three alphas followed by comma and space are ignored
 * . date accepted in format: mm/dd/yy, mm/dd/yyyy, dd-mmm-yy, dd-mmm-yyyy,
 *    dd mmm yy, dd mmm yyyy
 * . space or end of string required
 * . time accepted in format hh:mm:ss or hh:mm
 * . end of string accepted
 * . timezone accepted: hyphen followed by symbolic timezone, or space
 *    followed by signed numeric timezone or symbolic timezone
 * Examples of normal input:
 * . IMAP date-only (SEARCH): dd-mmm-yy, dd-mmm-yyyy, mm/dd/yy, mm/dd/yyyy
 * . IMAP date-time (INTERNALDATE):
 *    dd-mmm-yy hh:mm:ss-zzz
 *    dd-mmm-yyyy hh:mm:ss +zzzz
 * . RFC-822:
 *    www, dd mmm yy hh:mm:ss zzz
 *    www, dd mmm yyyy hh:mm:ss +zzzz
 */

#define MAILTMPLEN 512
#define BASEYEAR 1970

inline char *
ucase(char *s)
{
    char *t;
    /* if lowercase covert to upper */
    for (t = s; *t; t++) if (!(*t & 0x80) && islower(*t)) {
            *t = static_cast<char>(toupper(*t));
        }
    return s;                     /* return string */
}


struct p_date {
    unsigned int day;         /* day of month (1-31) */
    unsigned int month;       /* month of year (1-12) */
    unsigned int year;        /* year since BASEYEAR (expires in 127 yrs) */
    unsigned int hours;       /* hours (0-23) */
    unsigned int minutes;     /* minutes (0-59) */
    unsigned int seconds;     /* seconds (0-59) */
    unsigned int zoccident;   /* non-zero if west of UTC */
    unsigned int zhours;      /* hours from UTC (0-12) */
    unsigned int zminutes;    /* minutes (0-59) */
};

bool
mail_parse_date(p_date *elt,char *s)
{
    unsigned long d,m,y;
    int mi,ms;
    struct tm *t;
    time_t tn;
    char tmp[MAILTMPLEN];
    /* clear elt */
    elt->zoccident = elt->zhours = elt->zminutes =
                                       elt->hours = elt->minutes = elt->seconds =
                                               elt->day = elt->month = elt->year = 0;
    /* make a writeable uppercase copy */
    if (s && *s && (strlen(s) < static_cast<size_t>(MAILTMPLEN))) {
        s = ucase(strcpy(tmp,s));
    } else {
        return false;
    }
    /* skip over possible day of week */
    if (isalpha(*s) && isalpha(s[1]) && isalpha(s[2]) && (s[3] == ',') &&
            (s[4] == ' ')) {
        s += 5;
    }
    while (*s == ' ') {
        s++;    /* parse first number (probable month) */
    }
    if (!(m = strtoul(s,&s,10))) {
        return false;
    }
    switch (*s) {                 /* different parse based on delimiter */
        case '/':                     /* mm/dd/yy format */
            if (!((d = strtoul(++s,&s,10)) && *s == '/' &&
                    (y = strtoul(++s,&s,10)) && *s == '\0')) {
                return false;
            }
            break;
        case ' ':                     /* dd mmm yy format */
            while (s[1] == ' ') {
                s++;    /* slurp extra whitespace */
            }
        case '-':                     /* dd-mmm-yy format */
            d = m;                      /* so the number we got is a day */
            /* make sure string long enough! */
            if (strlen(s) < static_cast<size_t>(5)) {
                return false;
            }
            /* Some compilers don't allow `<<' and/or longs in case statements. */
            /* slurp up the month string */
            ms = ((s[1] - 'A') * 1024) + ((s[2] - 'A') * 32) + (s[3] - 'A');
            switch (ms) {               /* determine the month */
                case(('J'-'A') * 1024) + (('A'-'A') * 32) + ('N'-'A'):
                    m = 1;
                    break;
                case(('F'-'A') * 1024) + (('E'-'A') * 32) + ('B'-'A'):
                    m = 2;
                    break;
                case(('M'-'A') * 1024) + (('A'-'A') * 32) + ('R'-'A'):
                    m = 3;
                    break;
                case(('A'-'A') * 1024) + (('P'-'A') * 32) + ('R'-'A'):
                    m = 4;
                    break;
                case(('M'-'A') * 1024) + (('A'-'A') * 32) + ('Y'-'A'):
                    m = 5;
                    break;
                case(('J'-'A') * 1024) + (('U'-'A') * 32) + ('N'-'A'):
                    m = 6;
                    break;
                case(('J'-'A') * 1024) + (('U'-'A') * 32) + ('L'-'A'):
                    m = 7;
                    break;
                case(('A'-'A') * 1024) + (('U'-'A') * 32) + ('G'-'A'):
                    m = 8;
                    break;
                case(('S'-'A') * 1024) + (('E'-'A') * 32) + ('P'-'A'):
                    m = 9;
                    break;
                case(('O'-'A') * 1024) + (('C'-'A') * 32) + ('T'-'A'):
                    m = 10;
                    break;
                case(('N'-'A') * 1024) + (('O'-'A') * 32) + ('V'-'A'):
                    m = 11;
                    break;
                case(('D'-'A') * 1024) + (('E'-'A') * 32) + ('C'-'A'):
                    m = 12;
                    break;
                default:
                    return false;        /* unknown month */
            }
            if (s[4] == *s) {
                s += 5;    /* advance to year */
            } else {                    /* first three were OK, possibly full name */
                mi = *s;                  /* note delimiter, skip alphas */
                for (s += 4; isalpha(*s); s++) {
                    ;
                }
                /* error if delimiter not here */
                if (mi != *s++) {
                    return false;
                }
            }
            if ((y = strtoul(s,&s,10)) && (*s == '\0' || *s == ' ')) {
                break;    /* successfully parsed year */
            }
            return false;
        default:
            return false;          /* unknown date format */
    }
    /* minimal validity check of date */
    if ((d > 31) || (m > 12)) {
        return false;
    }
    /* two digit year */
    if (y < 100) {
        y += (y >= (BASEYEAR - 1900)) ? 1900 : 2000;
    }
    /* set values in elt */
    elt->day = static_cast<unsigned int>(d);
    elt->month = static_cast<unsigned int>(m);
    elt->year = static_cast<unsigned int>(y - BASEYEAR);
    ms = '\0';                    /* initially no time zone string */
    if (*s) {                     /* time specification present? */
        /* parse time */
        d = strtoul(s+1,&s,10);
        if (*s != ':') {
            return false;
        }
        ++s;
        m = strtoul(s,&s,10);
        y = (*s == ':') ? strtoul(s+1,&s,10) : 0;
        /* validity check time */
        if ((d > 23) || (m > 59) || (y > 59)) {
            return false;
        }
        /* set values in elt */
        elt->hours = static_cast<unsigned int>(d);
        elt->minutes = static_cast<unsigned int>(m);
        elt->seconds = static_cast<unsigned int>(y);
        switch (*s) {               /* time zone specifier? */
            case ' ':                   /* numeric time zone */
                while (s[1] == ' ') {
                    s++;    /* slurp extra whitespace */
                }
                if (!isalpha(s[1])) {     /* treat as '-' case if alphabetic */
                    /* test for sign character */
                    if ((elt->zoccident = (*++s == '-')) || (*s == '+')) {
                        s++;
                    }
                    /* validate proper timezone */
                    if (isdigit(*s) && isdigit(s[1]) && isdigit(s[2]) && isdigit(s[3])) {
                        elt->zhours = (*s - '0') * 10 + (s[1] - '0');
                        elt->zminutes = (s[2] - '0') * 10 + (s[3] - '0');
                    }
                    return true;               /* all done! */
                }
                /* falls through */
            case '-':                   /* symbolic time zone */
                if (!(ms = *++s)) {
                    ms = 'Z';
                } else if (*++s) {        /* multi-character? */
                    ms -= 'A';
                    ms *= 1024;  /* yes, make compressed three-byte form */
                    ms += ((*s++ - 'A') * 32);
                    if (*s) {
                        ms += *s++ - 'A';
                    }
                    if (*s) {
                        ms = '\0';    /* more than three characters */
                    }
                }
                break;
            default:                    /* ignore anything else */
                break;
        }
    }
    /*  This is not intended to be a comprehensive list of all possible
     * timezone strings.  Such a list would be impractical.  Rather, this
     * listing is intended to incorporate all military, North American, and
     * a few special cases such as Japan and the major European zone names,
     * such as what might be expected to be found in a Tenex format mailbox
     * and spewed from an IMAP server.  The trend is to migrate to numeric
     * timezones which lack the flavor but also the ambiguity of the names.
     *
     *  RFC-822 only recognizes UT, GMT, 1-letter military timezones, and the
     * 4 CONUS timezones and their summer time variants.  [Sorry, Canadian
     * Atlantic Provinces, Alaska, and Hawaii.]
     */
    switch (ms) {                 /* determine the timezone */
            /* Universal */
        case(('U'-'A')*1024)+(('T'-'A')*32):
        case(('U'-'A')*1024)+(('T'-'A')*32)+'C'-'A':
            /* Greenwich */
        case(('G'-'A')*1024)+(('M'-'A')*32)+'T'-'A':
        case 'Z':
            elt->zhours = 0;
            break;
            /* oriental (from Greenwich) timezones */
            /* Middle Europe */
        case(('M'-'A')*1024)+(('E'-'A')*32)+'T'-'A':
            /* British Summer */
        case(('B'-'A')*1024)+(('S'-'A')*32)+'T'-'A':
        case 'A':
            elt->zhours = 1;
            break;
            /* Eastern Europe */
        case(('E'-'A')*1024)+(('E'-'A')*32)+'T'-'A':
        case 'B':
            elt->zhours = 2;
            break;
        case 'C':
            elt->zhours = 3;
            break;
        case 'D':
            elt->zhours = 4;
            break;
        case 'E':
            elt->zhours = 5;
            break;
        case 'F':
            elt->zhours = 6;
            break;
        case 'G':
            elt->zhours = 7;
            break;
        case 'H':
            elt->zhours = 8;
            break;
            /* Japan */
        case(('J'-'A')*1024)+(('S'-'A')*32)+'T'-'A':
        case 'I':
            elt->zhours = 9;
            break;
        case 'K':
            elt->zhours = 10;
            break;
        case 'L':
            elt->zhours = 11;
            break;
        case 'M':
            elt->zhours = 12;
            break;
            /* occidental (from Greenwich) timezones */
        case 'N':
            elt->zoccident = 1;
            elt->zhours = 1;
            break;
        case 'O':
            elt->zoccident = 1;
            elt->zhours = 2;
            break;
        case(('A'-'A')*1024)+(('D'-'A')*32)+'T'-'A':
        case 'P':
            elt->zoccident = 1;
            elt->zhours = 3;
            break;
            /* Newfoundland */
        case(('N'-'A')*1024)+(('S'-'A')*32)+'T'-'A':
            elt->zoccident = 1;
            elt->zhours = 3;
            elt->zminutes = 30;
            break;
            /* Atlantic */
        case(('A'-'A')*1024)+(('S'-'A')*32)+'T'-'A':
            /* CONUS */
        case(('E'-'A')*1024)+(('D'-'A')*32)+'T'-'A':
        case 'Q':
            elt->zoccident = 1;
            elt->zhours = 4;
            break;
            /* Eastern */
        case(('E'-'A')*1024)+(('S'-'A')*32)+'T'-'A':
        case(('C'-'A')*1024)+(('D'-'A')*32)+'T'-'A':
        case 'R':
            elt->zoccident = 1;
            elt->zhours = 5;
            break;
            /* Central */
        case(('C'-'A')*1024)+(('S'-'A')*32)+'T'-'A':
        case(('M'-'A')*1024)+(('D'-'A')*32)+'T'-'A':
        case 'S':
            elt->zoccident = 1;
            elt->zhours = 6;
            break;
            /* Mountain */
        case(('M'-'A')*1024)+(('S'-'A')*32)+'T'-'A':
        case(('P'-'A')*1024)+(('D'-'A')*32)+'T'-'A':
        case 'T':
            elt->zoccident = 1;
            elt->zhours = 7;
            break;
            /* Pacific */
        case(('P'-'A')*1024)+(('S'-'A')*32)+'T'-'A':
        case(('Y'-'A')*1024)+(('D'-'A')*32)+'T'-'A':
        case 'U':
            elt->zoccident = 1;
            elt->zhours = 8;
            break;
            /* Yukon */
        case(('Y'-'A')*1024)+(('S'-'A')*32)+'T'-'A':
        case 'V':
            elt->zoccident = 1;
            elt->zhours = 9;
            break;
            /* Hawaii */
        case(('H'-'A')*1024)+(('S'-'A')*32)+'T'-'A':
        case 'W':
            elt->zoccident = 1;
            elt->zhours = 10;
            break;
            /* Nome/Bering/Samoa */
            //case (('N'-'A')*1024)+(('S'-'A')*32)+'T'-'A':
            //case (('B'-'A')*1024)+(('S'-'A')*32)+'T'-'A':
        case(('S'-'A')*1024)+(('S'-'A')*32)+'T'-'A':
        case 'X':
            elt->zoccident = 1;
            elt->zhours = 11;
            break;
        case 'Y':
            elt->zoccident = 1;
            elt->zhours = 12;
            break;
        default:                      /* unknown time zones treated as local */
            tn = time(0);               /* time now... */
            t = localtime(&tn);         /* get local minutes since midnight */
            mi = t->tm_hour * 60 + t->tm_min;
            ms = t->tm_yday;            /* note Julian day */
            if ((t = gmtime(&tn))) {     /* minus UTC minutes since midnight */  // Here we have '=' assignment
                mi -= t->tm_hour * 60 + t->tm_min;
                /* ms can be one of:
                 *  36x  local time is December 31, UTC is January 1, offset -24 hours
                 *    1  local time is 1 day ahead of UTC, offset +24 hours
                 *    0  local time is same day as UTC, no offset
                 *   -1  local time is 1 day behind UTC, offset -24 hours
                 * -36x  local time is January 1, UTC is December 31, offset +24 hours
                 */
                if (ms -= t->tm_yday) {   /* correct offset if different Julian day */
                    mi += ((ms < 0) == (abs(ms) == 1)) ? -24*60 : 24*60;
                }
                if (mi < 0) {             /* occidental? */
                    mi = abs(mi);           /* yup, make positive number */
                    elt->zoccident = 1;     /* and note west of UTC */
                }
                elt->zhours = mi / 60;    /* now break into hours and minutes */
                elt->zminutes = mi % 60;
            }
            break;
    }
    return true;
}

namespace {

inline time_t make_time_t(struct tm* tm_time)
{
#ifdef ARCADIA_BUILD
    return TimeGM(tm_time);
#else
    return timegm(tm_time);
#endif
}

} // namespace

namespace rfc2822 {

rfc2822date::rfc2822date(const string &date_time)
{
    p_date elt;
    good_parse = mail_parse_date(&elt, const_cast<char*>(date_time.c_str()));
    memset(&tm_time, 0, sizeof(tm_time));
    // // Translate to GMT time
    tm_time.tm_sec  = elt.seconds;
    tm_time.tm_min  = elt.minutes + (elt.zoccident ? 1 : -1) * 60 * elt.zhours;
    tm_time.tm_hour = elt.hours;
    tm_time.tm_mday = elt.day;
    tm_time.tm_mon  = elt.month - 1;
    tm_time.tm_year = elt.year - 1900 + BASEYEAR;
    tm_gmtoff = static_cast<long>(elt.zoccident ? -1 : 1) * (static_cast<long>(elt.zhours)
        * 3600 + static_cast<long>(elt.zminutes) * 60);
    tt_cache = 0;
}

time_t rfc2822date::unixtime()
{
    if (!good_parse) {
        return 0;
    }
    return tt_cache == 0 ? tt_cache = make_time_t(&tm_time) : tt_cache;
}

const struct tm *rfc2822date::structime() {
    if (!good_parse) {
        return NULL;
    }
    return &tm_time;
}

long rfc2822date::offset() const {
    if (!good_parse) {
        return 0;
    }
    return tm_gmtoff;
}

#ifdef RFC2822DATEDEBUG
void rfc2822date::debug()
{
    if (good_parse) {
        std::cout << asctime(&tm_time);
        std::cout << ((float) tm_gmtoff)/3600 << std::endl;
    } else {
        std::cerr << "Didn't parsed successfully" << std::endl;
    }
}

#endif
}
