import argparse
import datetime
import json
import os
import requests
import sys
import urllib3
from startrek_client import Startrek
from tp_api_client import tp_api_client

urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)

CHECKBOX = '<# <input type="checkbox"> #>'


def handle_options():
    parser = argparse.ArgumentParser()
    parser.add_argument("-p", "--project", dest="project")
    return parser


def get_project(args):
    if args.project and args.project in ['iOS', 'Android']:
        return args.project
    else:
        output_in_console('Project is not specified or project name incorrect (correct name: iOS or Android)')
        sys.exit()


def welcome_message():
    output_in_console('''
    \x1b[1m
    Hi!
    This is script for create tickets for version!
    Let's go! ;)\x1b[0m''')


def output_in_console(string):
    print("{date}   {string}".format(date=datetime.datetime.now(), string=string))


def input_from_console(string):
    data = input("{date}   {string}".format(date=datetime.datetime.now(), string=string))
    return data


def stop_script(string):
    output_in_console('\x1b{}\x1b[0m'.format(string))
    sys.exit(0)


def check_return_code(request):
    status_OK = [200, 201]

    if not status_OK.count(request.status_code):
        output_in_console('\033[91m Error\033[00m in request. Status code = {}, reason = {}'
                          .format(request.status_code, request.reason))


class StartTrack:
    def __init__(self, data):
        self.client = Startrek(base_url='https://st-api.yandex-team.ru',
                               useragent=data['requester'],
                               token=data['AUTH_ST'])
        self.token = data['AUTH_ST']

        self.queue = str(data['queue'])
        self.beta = data['beta']
        self.release_type = str(data['release_type'])
        self.checklist = data['checklist']
        self.dashboard = data['dashboard_filters']

        self.platform = str(data['platform'])
        self.version = str(data['version_app'])
        self.smoke_case = data['smoke_case']

        self.id_version = self.get_id_version()
        self.description_for_testing_ticket = "No description"
        self.task_key = str(data['release_ticket']) if data['release_ticket'] else None

    def create_version_startrek(self):
        version = self.client.versions.create(queue=self.queue, name='{} {}'.format(self.platform, self.version))
        self.id_version = version.id
        output_in_console('Version in StarTrack was created.')

    def get_description_for_testing_ticket(self, data_json):
        all_ticket = self.client.filters.create(name='{} - All tickets'.format(self.version),
                                                query=data_json['all_ticket'].replace('[version]', self.version))

        found_bugs = self.client.filters.create(name='{} - Found bugs'.format(self.version),
                                                query=data_json['found_bugs'].replace('[version]', self.version))

        backlog_bug = self.client.filters.create(name='{} - Backlog bugs'.format(self.version),
                                                 query=data_json['backlog_bug'].replace('[version]', self.version))

        backlog_task = self.client.filters.create(name='{} - Backlog task'.format(self.version),
                                                  query=data_json['backlog_task'].replace('[version]', self.version))

        url_testpalm = 'Version in TestPalm  {testpalm}/{platform}{version}'.format(testpalm=data_json['testpalm'],
                                                                                    version=self.version,
                                                                                    platform=self.platform)

        autotest = 'Autotest {}_autotest\n'.format(url_testpalm) if data_json['create_version_autotest'] else ''

        description = \
            'Link on dashboard {dashboard}\n\n' \
            'All tickets was fixed in version {version} https://st.yandex-team.ru/filters/filter:{all_ticket}\n' \
            'The bugs found while testing version {version} https://st.yandex-team.ru/filters/filter:{found_bugs}\n\n' \
            'Bugs from backlog was fixed in version {version} https://st.yandex-team.ru/filters/filter:{backlog_bug}\n' \
            'Tasks from backlog was fixed in version {version} https://st.yandex-team.ru/filters/filter:{backlog_task}\n\n' \
            '{url_testpalm}\n' \
            '{autotest_testpalm}\n'.format(dashboard=data_json['dashboard'], all_ticket=all_ticket.id,
                                           backlog_bug=backlog_bug.id, backlog_task=backlog_task.id,
                                           version=self.version, found_bugs=found_bugs.id,
                                           url_testpalm=url_testpalm, autotest_testpalm=autotest)

        return description

    def get_id_version(self):
        all_version = self.client.queues[self.queue].versions
        for item in all_version:
            if item.name == '{} {}'.format(self.platform, self.version):
                self.id_version = item.id
                return self.id_version
        output_in_console("\x1b[91m Id version wasn't found.\x1b[0m Please, add Affected and Fix version by manual")
        return ''

    def get_id_component(self):
        all_components = self.client.queues[self.queue].components
        for item in all_components:
            if item.name == '{}'.format(self.platform):
                return item.id
        output_in_console("\x1b[91m Id components wasn't found.\x1b[0m Please, add Components by manual")
        return ''

    def change_filter_for_dasboadrd(self):
        for number_filter in self.dashboard:
            try:
                filter_st = self.client.filters.get(number_filter['filter'])
                old_version = filter_st.name.split(":")[0]
                new_name = filter_st.name.replace(old_version, ".".join(self.version.split(".")[0:2]))
                new_query = filter_st.query.replace(old_version, ".".join(self.version.split(".")[0:2]))
                self.client.filters.update(filter_st, name=new_name, query=new_query)
            except:
                continue

    def create_checklist(self, key, checklist):
        headers_startrek = {
            'Authorization': f'OAuth {self.token}',
            'Content-Type': 'application/json',
            'User-Agent': 'curl/7.53.1',
            'Connection': 'close',
            'Accept': '*/*'}

        data = [{"checked": False, "text": item} for item in list(checklist)]

        r = requests.post(f'https://st-api.yandex-team.ru/v2/issues/{key}/checklistItems',
                          data=json.dumps(data),
                          headers=headers_startrek)
        check_return_code(r)
        output_in_console("Check list for release ticket was created")

    def create_ticket_for_testing(self, data_json, ):
        release_type = input_from_console(
            '\x1b[1m\n Release type is "{}". Is it correct? (tap \'Enter\' for continue; enter release type [Feature, '
            'Technical, Hotfix, Out of turn] for change: \x1b[0m'.format(self.release_type))

        release_type = release_type if release_type else self.release_type
        self.description_for_testing_ticket = self.get_description_for_testing_ticket(data_json)
        id_component = self.get_id_component()

        ticket = self.client.issues.create(queue=self.queue,
                                           summary='{} {} testing'.format(self.platform, self.version),
                                           description=self.description_for_testing_ticket,
                                           type='release',
                                           affectedVersions=self.id_version,
                                           fixVersions=self.id_version,
                                           components=id_component,
                                           releaseType=release_type,
                                           qa=data_json['requester'])
        self.task_key = ticket.key
        self.create_checklist(self.task_key, self.checklist)
        output_in_console("Ticket for testing was created in Startrek: \x1b[1m {} \x1b[0m".format(self.task_key))

    #    self.create_ticket_in_our_queue(summary='Smoke scenario', description=CHECKBOX + f'\n\n{CHECKBOX}'.join(self.smoke_case))

    def create_task_texts(self, summary, description, queue):
        task = self.client.issues.create(queue=queue,
                                         summary=summary,
                                         description=description,
                                         type='task')

        testing_ticket = self.client.issues[self.task_key]
        testing_ticket.comments.create(text='Create ticket for release notes: {}'.format(task.key))
        output_in_console("Tickets on release notes '{}' was created in Startrek: {}".format(summary, task.key))

    def create_ticket_in_our_queue(self, summary, description, checklist=None):
        id_version = self.get_id_version()
        id_component = self.get_id_component()
        task_key = self.client.issues.create(queue=self.queue,
                                             summary="[{}] {}".format(self.version, summary),
                                             description=description,
                                             type='task',
                                             affectedVersions=id_version,
                                             fixVersions=id_version,
                                             components=id_component)
        if checklist is not None:
            self.create_checklist(task_key.key, checklist)
        testing_ticket = self.client.issues[self.task_key]
        testing_ticket.comments.create(text='Create ticket: {}'.format(task_key.key))
        output_in_console("Ticket '{}' for testing was created in Startrek: {}".format(summary, task_key.key))


class TestPalm:
    def __init__(self, data, st):
        self.project = data['testpalm_project']
        self.client = tp_api_client.TestPalmClient(auth=data['AUTH_TP'])
        self.platform = st.platform
        self.version = st.version
        self.ticket_for_testing = st.task_key
        self.suites = data['suites']
        self.queue = data['queue']

    def create_testpalm_version(self, title, description):
        data = {
            'id': title,
            'description': description
        }

        self.client.create_version(project=self.project, data=data)
        output_in_console("Version {} in Testpalm  was created".format(title))

    def create_first_runs(self):
        output_in_console("Create runs....")

        for item in self.suites:
            self.create_testrun(item, '{version} {name} {enviroment}'.format(version=self.version,
                                                                             name=item['run'],
                                                                             enviroment=item['enviroment'].strip()))

            output_in_console('Run \'{name} {enviroment}\' was created'.format(name=item['run'],
                                                                               enviroment=item['enviroment']))

    def create_testrun(self, suite, title):
        data = {
            'title': title,
            'version': '%s%s' % (self.platform, self.version),
            'environments': [{
                'title': suite['enviroment'],
                'description': suite['enviroment'],
                'default': False
            }],
            'testSuite': {
                'id': str(suite['key'])
            },
            'parentIssue': {
                'assignee': [],
                'createdTime': 0,
                'display': self.ticket_for_testing,
                'groupId': self.queue,
                'id': self.ticket_for_testing,
                'isBug': False,
                'url': 'https://st.yandex-team.ru/{}'.format(self.ticket_for_testing),
                'status': {
                    'id': 'open',
                    'title': 'Открыт'
                },
                'isResolved': False,
                'title': self.queue,
                'trackerId': 'Startrek'
            }
        }
        self.client.create_testrun_from_suite(project=self.project, data=data)


class Scenario:
    def __init__(self):
        self.testing_version = False
        self.ticket_text = False
        self.ticket_text_edit = False
        self.ticket_translate = False
        self.ticket_libraries = False
        self.ticket_react = False
        self.ticket_telemost = False
        self.version_st = False


def main():
    args = handle_options().parse_args()
    project = get_project(args)
    urllib3.disable_warnings()
    welcome_message()
    scenario = input_from_console("Please, choose scenario for running script: \n"
                                  "\t 0) Create version in StartTrack (exp: 'Android 4.4.0', 'iOS 3.83.0');\n"
                                  "\t 1) Create ticket for testing and version in TestPalm;\n"
                                  "\t 2) Update dashboard for new new release;\n"
                                  "\t 3) Create ticket for release notes in our queue;\n"
                                  "\t 4) Create ticket for updating libraries;\n"
                                  "\t 5) Create ticket for updating React;\n"
                                  "\t 6) Create ticket for updating Telemost;\n"
                                  "\t 7) Exit \n"
                                  "\n\tScenario (enter numbers by spaces): ")

    create = Scenario()
    create.version_st = True if scenario.count('0') else False
    create.testing_version = True if scenario.count('1') else False
    create.update_dashboard = True if scenario.count('2') else False
    create.ticket_text = True if scenario.count('3') else False
    create.ticket_libraries = True if scenario.count('4') else False
    create.ticket_react = True if scenario.count('5') else False
    create.ticket_telemost = True if scenario.count('6') else False

    if scenario.count('7'):
        stop_script('Script was stopped by user')

    with open(os.path.join(os.path.dirname(__file__), f'variable_{project}.json')) as file:
        data = json.load(file)

    st = StartTrack(data)

    if create.version_st:
        st.create_version_startrek()

    if create.testing_version:
        incorrect_data = input_from_console(
            '\x1b[1m\n\nCheck input data for create tickets and versions:\n'
            '\t App version = {version}\n'
            '\t Is data correct? (tap \'Enter\' for continue; \'No\' for break)\x1b[0m\n'.format(version=st.version))

        if incorrect_data:
            stop_script('Running of script was stopped by user')

        st.create_ticket_for_testing(data)

        tp = TestPalm(data, st)
        tp.create_testpalm_version(title='{}{}'.format(tp.platform, tp.version),
                                   description=st.description_for_testing_ticket)
        if data['create_version_autotest']:
            tp.create_testpalm_version(title='{}{}_autotest'.format(tp.platform, tp.version),
                                       description="")
        tp.create_first_runs()

    if create.update_dashboard:
        st.change_filter_for_dasboadrd()

    if create.ticket_libraries:
        if project == 'iOS':
            st.create_ticket_in_our_queue(
                summary='Update 3rd party libraries',
                description='',
                checklist=data['update_libraries_checklist'])
        else:
            st.create_ticket_in_our_queue(summary='Ticket for update libraries', description='')

    if create.ticket_text:
        st.create_ticket_in_our_queue \
            (summary='Ticket for release notes',
             description='**((https://st.yandex-team.ru/createTicket?queue=EDIT&_form=15297 Заказать текст для Release Notes))\n'
                         '((https://st.yandex-team.ru/createTicket?queue=TRANSLATE&_form=13067 Перевести текст Release Notes))**')

    if create.ticket_react:
        st.create_ticket_in_our_queue(summary='Ticket for update React', description='')

    if create.ticket_telemost:
        st.create_ticket_in_our_queue(summary='Ticket for update Telemost', description='')

    output_in_console('\x1b[1mScript finished successfully!\x1b[0m')


if __name__ == '__main__':
    main()
