from enum import Enum
import random

import urllib3
import warnings
from startrek_client import Startrek

from sklearn.linear_model import Lasso
from sklearn.preprocessing import PolynomialFeatures

import sklearn.metrics as metrics
import matplotlib.pyplot as plt


class Priority(Enum):
    MINOR = 'minor'
    NORMAL = 'normal'
    CRIT = 'crit'


class Weight(Enum):
    MINOR = 100
    NORMAL = 1000
    CRIT = 10000


class Ticket:
    def __init__(self, ticket):
        bug_type = {'Блокирует сценарий': 2,
                    'Баг безопасности': 4,
                    'Линейный баг': 1,
                    'UI баг': 1,
                    'Специфичный кейс': 0.5,
                    'Не заметен пользователю': 0.5,
                    'Можно обойти': 0.5,
                    'Метрика': 0.5,
                    '-': None}

        self.crash = 2 if ticket.crash == 'Yes' else 0
        self.audience = float(ticket.audience)
        self.duplicatesCount = 0 if not ticket.duplicatesCount else int(ticket.duplicatesCount)
        self.type = bug_type[ticket.bugType] if ticket.bugType else None
        self.reproduce = 0 if ticket.reproducibility == 'Периодически' else 2
        self.weight = int(ticket.weightOne)


class User(Enum):
    TOKEN_ST = 'AQAD-***-**'
    NAME = 'robot-mobmailstat'


issues = [
    'MOBILEMAIL-8207', 'MOBILEMAIL-11727', 'MOBILEMAIL-11452', 'MOBILEMAIL-11005',
    'MOBILEMAIL-10457', 'MOBILEMAIL-12316', 'MOBILEMAIL-11721', 'MOBILEMAIL-12206',
    'MOBILEMAIL-12782', 'MOBILEMAIL-12775', 'MOBILEMAIL-12762', 'MOBILEMAIL-12700',
    'MOBILEMAIL-12667', 'MOBILEMAIL-12663', 'MOBILEMAIL-12649', 'MOBILEMAIL-12621',
    'MOBILEMAIL-12576', 'MOBILEMAIL-12547', 'MOBILEMAIL-12512', 'MOBILEMAIL-12501',
    'MOBILEMAIL-12456', 'MOBILEMAIL-12442', 'MOBILEMAIL-12372', 'MOBILEMAIL-11361',
    'MOBILEMAIL-12369', 'MOBILEMAIL-12255', 'MOBILEMAIL-12178', 'MOBILEMAIL-588',
    'MOBILEMAIL-11253', 'MOBILEMAIL-10832', 'MOBILEMAIL-10679', 'MOBILEMAIL-8425',
    'MOBILEMAIL-10380', 'MOBILEMAIL-10293', 'MOBILEMAIL-9971', 'MOBILEMAIL-8737',
    'MOBILEMAIL-9763', 'MOBILEMAIL-9068', 'MOBILEMAIL-9025', 'MOBILEMAIL-8854',
    'MOBILEMAIL-7712', 'MOBILEMAIL-7208', 'MOBILEMAIL-7193', 'MOBILEMAIL-12777',
    'MOBILEMAIL-6799', 'MOBILEMAIL-6201', 'MOBILEMAIL-2623', 'MOBILEMAIL-2449',
    'MOBILEMAIL-12567', 'MOBILEMAIL-12457', 'MOBILEMAIL-12427', 'MOBILEMAIL-12376'
]


def get_priority(weight):
    if weight < Weight.MINOR.value:
        return Priority.MINOR

    if weight < Weight.NORMAL.value:
        return Priority.NORMAL

    return Priority.CRIT


def get_color(weight):
    priority = get_priority(weight)
    if priority == Priority.MINOR:
        return 'blue'
    if priority == Priority.NORMAL:
        return 'green'
    return 'red'


def bingo(weight, weight_prediction):
    priority = get_priority(weight)
    priority_prediction = get_priority(weight_prediction)

    if priority_prediction == priority:
        return priority, True

    return priority, False


def prediction_and_get_formula(input_data, test_data, degree, alpha, write_formula_in_file=False):
    # print("\n---------Polynomial Features---------\n")

    lenght = 0
    data = []
    result_function = []
    result = dict(len=0, prediction_result=0)

    for ticket in input_data:
        data.append([ticket.crash, ticket.audience, ticket.duplicatesCount, ticket.type, ticket.reproduce])
        result_function.append(ticket.weight)

    poly = PolynomialFeatures(degree=degree)
    data_ = poly.fit_transform(data)

    clf = Lasso(alpha=alpha)

    clf.fit(data_, result_function)

    function = "F U N C T I O N: \n\n"
    coef = clf.coef_
    name = poly.get_feature_names()

    for i in range(0, len(name)):
        if coef[i]:
            lenght += 1
            data = '\n\t{} {},'.format(round(float(coef[i]), 5), name[i])
            data = data.replace('^2', '**2')
            data = data.replace('^3', '**3')
            function += data.replace(' ', ' * ')


    if write_formula_in_file:
        f = open('function degree = {} alpha = {}.txt'.format(degree, alpha), 'w')
        f.write('degree = {:1g}\t\talpha = {:2g}\t\tlen = {:3g}\n\n'.format(degree, alpha, lenght))
        f.write('x0 = Crash     x1 = Audience       x2 = DuplicatesCount    x3 = Type    x4 = Reproduce')
        function = function.replace('x0', 'self.crash')
        function = function.replace('x1', 'audience')
        function = function.replace('x2', 'self.numberOfComplaints')
        function = function.replace('x3', 'self.type')
        function = function.replace('x4', 'self.reproduce')
        f.write("{}\n\n".format(function))
        prediction = clf.predict(data_)
        for i in range(0, len(result_function)):
            f.write('ticket = {}\t\tweight = {}\t\tweight_prediction = {}\n'.format(issues[i], result_function[i], prediction[i]))
        f.close()

    # print(clf.score(x_, y))

    result['len'] = lenght

    if test_data:
        data = [[test_data.crash, test_data.audience, test_data.duplicatesCount, test_data.type, test_data.reproduce]]
        data_ = poly.fit_transform(data)
        predict = clf.predict(data_)
        result['y_test'] = predict

    return result


def my_score(y, y_test):
    sum = 0
    type = dict(minor=0, normal=0, crit=0)
    score = dict(minor=0, normal=0, crit=0)
    for i in range(0, len(y)):
        type_i, flag = bingo(y[i], y_test[i])
        type[type_i.value] += 1
        score[type_i.value] += int(flag)
        sum += int(flag)

    return float(sum / len(y)), \
           float(score['minor'] / type['minor']), \
           float(score['normal'] / type['normal']), \
           float(score['crit'] / type['crit'])


def plot(data):
    N = 60
    x = []
    y = []
    z = []
    col = []

    for item in data:
        x.append(item[1])
        y.append(item[3])
        z.append(item[1])
        col.append(get_color(item[-1]))

    data = (x, y, z)

    # Create plot
    fig = plt.figure()
    ax = fig.add_subplot(111, projection='3d')

    for aud, type, color, crash in zip(x, y, col, z):
        ax.scatter(aud, type, crash, alpha=0.8, c=color, edgecolors='none', s=30)

    ax.set_xlabel('X Audi')
    ax.set_ylabel('Y Type')
    ax.set_zlabel('Z Weight')

    plt.title('Matplot scatter plot')
    plt.legend(loc=2)
    plt.show()


def get_weight_by_priority(priority):
    random.seed(1)
    if priority == Priority.MINOR:
        return random.randint(1, Weight.MINOR.value - 1)

    if priority == Priority.NORMAL:
        return random.randint(Weight.MINOR, Weight.NORMAL.value - 1)

    return random.randint(Weight.NORMAL.value, Weight.CRIT.value - 1)


def get_data():
    urllib3.disable_warnings()
    client = Startrek(base_url='https://st-api.yandex-team.ru',
                      useragent=User.NAME.value,
                      token=User.TOKEN_ST.value)

    data = []
    i = 0
    for iss in issues:
        ticket = client.issues[iss]
        data.append(Ticket(ticket))

    with open("data.txt", 'w') as f:
        for ticket in data:
            f.write("{}\t Crash={}\t Audience={}\t DuplicatesCount={}\t Type={}\t Reproduce = {}\n"
                    .format(issues[i], ticket.crash, ticket.audience, ticket.duplicatesCount, ticket.type, ticket.reproduce))
            i += 1

    return data


def main():
    warnings.filterwarnings("ignore")
    data = get_data()
    arr_degrees = [2, 3]
    arr_alphes = [1, 5, 10, 50, 100, 500]

    f = open('Summary.txt', 'w')

    for degree in arr_degrees:
        for alpha in arr_alphes:
            all_result = dict(len=0, y_test=[], y=[])
            for i in range(0, len(data)):
                test = data.pop(i)
                res = prediction_and_get_formula(data, test, degree, alpha)
                data.insert(i, test)

                all_result['len'] += res['len']
                all_result['y_test'].append(int(res['y_test'][0]))
                all_result['y'].append(int(test.weight))

            score = metrics.r2_score(all_result['y'], all_result['y_test'])
            my_all_sc, sc_minor, sc_normal, sc_crit = my_score(all_result['y'], all_result['y_test'])

            # with open('{} {}.txt'.format(degree, alpha), 'w') as f_y:
            #     y = all_result['y']
            #     y_pre = all_result['y_test']
            #     for i in range(0, len(y)):
            #         f_y.write('y = {}\t\t\ty_pre = {}\n'.format(y[i], y_pre[i]))

            f.write('degree = {:1g}\t\talpha = {:2g}\t\tscore = {:.4}\t\tlen = {:3g}\t\t'
                    'my_score = {:.1%}\t\tminor = {:.1%}\t\tnormal = {:.1%}\t\tcrit = {:.1%}\n'
                    .format(degree,
                            alpha,
                            score,
                            float(all_result['len'] / len(data)),
                            my_all_sc, sc_minor, sc_normal, sc_crit))

    f.close()
    prediction_and_get_formula(data, test_data=None, degree=2, alpha=1, write_formula_in_file=True)
    prediction_and_get_formula(data, test_data=None, degree=2, alpha=5, write_formula_in_file=True)
    prediction_and_get_formula(data, test_data=None, degree=2, alpha=10, write_formula_in_file=True)
    prediction_and_get_formula(data, test_data=None, degree=2, alpha=50, write_formula_in_file=True)
    prediction_and_get_formula(data, test_data=None, degree=2, alpha=100, write_formula_in_file=True)
    prediction_and_get_formula(data, test_data=None, degree=2, alpha=500, write_formula_in_file=True)
    prediction_and_get_formula(data, test_data=None, degree=3, alpha=500, write_formula_in_file=True)
    # plot(data)


if __name__ == "__main__":
    main()
