# Example run : python3 pony_2.py

# https://api.stat.yandex-team.ru/_api/statreport/json/Mail/Totals/Mail_Mobile/EventAudience2?_format=json&_period_distance=30&device_type=_total_&event_name=_total_&os_major_version=_total_&platform=android&scale=d&session_type=_total_&type=json
# https://github.yandex-team.ru/statbox/python-statinfra/blob/master/docs/statface_client.md

import argparse
from dataclasses import dataclass
import os
import time
import urllib3
import logging

import statface_client
from startrek_client import Startrek

logging.basicConfig(format='[%(asctime)s | %(levelname)s]: %(message)s',
                    datefmt='%m.%d.%Y %H:%M:%S',
                    level=logging.INFO)

@dataclass
class User:
    TOKEN_ST = os.environ['OAUTH_TOKEN_STARTREK']
    TOKEN_STAT = os.environ['OAUTH_TOKEN_STAT']
    NAME = os.environ['USER_YANDEX']
    APP = 'mail'


@dataclass
class Filter:
    all = 'Queue: "Yandex Mobile Mail" AND Type: Bug AND Tags: inProd  AND (Tags: event_add OR Tags: event_skip ) AND Status: Open, "Need Info", Later'
    without_weight = 'Queue: "Yandex Mobile Mail" AND Type: Bug AND Tags: inProd  AND (Tags: event_add OR Tags: event_skip ) AND Status: !Closed AND ("Weight 1": empty() OR "Weight 1": < 0)'
    support = 'Queue: "Yandex Mobile Mail" AND Type: Bug AND Tags: inProd  AND (Tags: event_add OR Tags: event_skip ) AND "Duplicates Count": changed( date: today() )'
    only_new = 'Queue: "Yandex Mobile Mail" AND Type: Bug AND Tags: inProd  AND (Tags: event_add OR Tags: event_skip ) AND Created: today()'


@dataclass
class PonyWorld:
    stop_word = 'event_skip'
    done = 'pony_done_prioritization'


@dataclass
class Priority:
    minor = 'minor'
    normal = 'normal'
    critical = 'critical'
    blocker = 'blocker'


urllib3.disable_warnings()
client = Startrek(base_url='https://st-api.yandex-team.ru',
                  useragent=User.NAME,
                  token=User.TOKEN_ST)

summonees = ['a-zoshchuk', 'al-charnyshou', 'chibert', 'fanem']

ticket = ['MOBILEMAIL-13372']


class Prioritization:
    def __init__(self, flags):
        self.all_priority = {Priority.minor: 100, Priority.normal: 1000, Priority.critical: 10000,
                             Priority.blocker: 100000}
        self.all_crash = {'No': 0, 'Yes': 2, 'Нет': 0, 'Да': 2, '-': None}
        self.bug_type = {'Блокирует сценарий': 2,
                         'Баг безопасности': 4,
                         'Линейный баг': 1,
                         'UI баг': 1,
                         'Специфичный кейс': 0.1,
                         'Не заметен пользователю': 0.5,
                         'Можно обойти': 0.5,
                         'Метрика': 0.5,
                         '-': None}

        self.is_update_priotity = False if flags.not_priority else True
        self.is_update_weight = False if flags.not_weight else True
        self.is_update_audience = False if flags.not_audience else True
        self.is_list_ticket = flags.ticket

        self.filter = Filter.support if flags.filter_support else Filter.all
        self.filter = Filter.only_new if flags.filter_only_new else self.filter
        self.filter = Filter.without_weight if flags.filter_without_weight else self.filter

        self.all_event = []
        self.ticket_change_priority = []

    def add_ticket_in_list_change_priority(self, ticket: str, text: str) -> None:
        self.ticket_change_priority.append(dict(key=ticket, text=text))

    def create_comments(self) -> None:
        for item in self.ticket_change_priority:
            new_revision = client.issues[item['key']]
            new_revision.comments.create(text=item['text'], summonees=summonees)
            logging.info(f"Invite in ticket {item['key']}")


class Audience:
    def __init__(self):
        self.client = statface_client.StatfaceClient(oauth_token=User.TOKEN_STAT,
                                                     host=statface_client.STATFACE_PRODUCTION)
        logging.info("Client StatFace was created.")
        self.report = self.client.get_report('Mail/Totals/Mail_Mobile/EventAudience3')

    def get_all_event(self, platform: str) -> list:
        data = self.report.download_data(scale='daily',
                                         application=User.APP,
                                         _period_distance=30,
                                         device_type='_total_',
                                         os_major_version='_total_',
                                         platform=platform,
                                         session_type='_total_',
                                         type='json')

        logging.info("Data from report was gotten.")

        if not len(data):
            logging.info(f"Data from report ia empty. All event for '{platform}' wasn't gotten.")
            return 0

        event = [item['event_name'] for item in data]

        logging.info(f"Count of different event for '{platform}' is {len(event)}")
        return event

    def get_audience_by_event(self, event: str, platform: str) -> float:
        data = self.report.download_data(scale='daily',
                                         application=User.APP,
                                         _period_distance=30,
                                         device_type='_total_',
                                         event_name=event,
                                         os_major_version='_total_',
                                         platform=platform,
                                         session_type='_total_',
                                         type='json')

        logging.info("Data from report was gotten.")

        if not len(data):
            logging.info(
                f"Data from report is empty. There isn't action '{event}' for '{platform}' during 30 days.Audience = 0%")
            return 0

        average = 0
        for item in data:
            average += item['uuids_count']

        average = float(average / len(data))
        logging.info(f"Average count of '{event}' for '{platform}' == {average}.")
        return average

    def get_audience(self, event_name: str, platform: str) -> float:
        total = self.get_audience_by_event('_total_', platform)
        event = self.get_audience_by_event(event_name, platform)
        event_audience = float(event / total)

        percent_audience = (round(event_audience * 100, 2))
        logging.info(f"Percent audience of '{event}' == {percent_audience}.")
        return percent_audience


class Ticket:
    def __init__(self, item, prioritization: Prioritization):
        self.is_update_audience = True
        self.key = item.key

        self.platform = self.get_platform_in_ticket([component.name for component in item.components])
        self.event_name = item.eventFromMetrica if item.eventFromMetrica else self.get_event_in_ticket(item.tags,
                                                                                                       prioritization.all_event,
                                                                                                       PonyWorld.stop_word)

        self.priority = item.priority
        self.audience = item.audience
        self.weight = item.weightOne
        self.type = prioritization.bug_type[item.bugType] if item.bugType else None
        self.crash = prioritization.all_crash[item.crash] if item.crash else None
        self.is_crash_from_metrica = True if (item.bugType == 'Метрика') and self.crash else False
        self.duplicatesCount = 0 if not item.duplicatesCount else int(item.duplicatesCount)
        self.reproduce = 1 if item.howReproduce == '100%' else 0.4

    def get_type(self, max_weight_of_priority: dict) -> str:
        if self.weight < max_weight_of_priority[Priority.minor]:
            return Priority.minor
        if self.weight < max_weight_of_priority[Priority.normal]:
            return Priority.normal
        if self.weight < max_weight_of_priority[Priority.critical]:
            return Priority.critical

        return Priority.blocker

    def get_platform_in_ticket(self, arr_components: list):
        android = 'android'
        ios = 'iOS'

        # for component in arr_components:
        #     if component.name.lower() == android.lower():
        #         return android
        #     elif component.name.lower() == ios.lower():
        #         return ios

        logging.info(f"Components={arr_components}")

        if android.capitalize() in arr_components:
            return android

        if ios in arr_components:
            return ios

        logging.info(f'Platform wasn\'t found. Components = {[i for i in arr_components]}')
        self.is_update_audience = False
        return None

    def get_event_in_ticket(self, tags: list, event: list, stop_word: str):
        if stop_word in tags:
            logging.info("Stop word was found in ticket.")
            self.is_update_audience = False
            return None

        event_tag = list(set(tags) & set(event))
        if len(event_tag):
            logging.info(f"Event was found. Event = {event_tag[0]}")
            return event_tag[0]

        logging.info(f"Event wasn't found. Tags = {tags}")
        self.is_update_audience = False
        return None

    def get_weight_for_ticket(self, audience, min_audience=0.01) -> int:
        if (audience is None) or (self.type is None) or (self.crash is None):
            logging.info(f"There isn't any field for ticket {self.key}. "
                         f"Audience = {audience}, Crash = {self.crash}, Bug Type = {self.type}.")
            return -1

        addend = [12.42529 * audience,
                  57.81038 * self.crash * audience,
                  104.67725 * self.crash * self.type,
                  0.34347 * audience ** 2,
                  # 4.0223 * audience * self.duplicatesCount,
                  -6.64942 * audience * self.type,
                  0.72038 * audience * self.reproduce,
                  # -0.49409 * self.duplicatesCount ** 2,
                  # 86.33953 * self.duplicatesCount * self.reproduce,
                  55.59485 * self.type ** 2]

        self.weight = round(sum(addend))
        self.weight = self.weight + 50*self.duplicatesCount

        if self.weight < 0:
            return 0

        if audience < min_audience:
            self.weight = round(self.weight / (min_audience / audience))

        if self.is_crash_from_metrica:
            self.weight = min(99, self.weight)

        return self.weight

    def update_priority(self, prioritization: Prioritization) -> Prioritization:
        priority_by_weight = self.get_type(prioritization.all_priority).lower()
        add_in_change_list = prioritization.add_ticket_in_list_change_priority

        if self.priority.key.lower() != priority_by_weight:
            if priority_by_weight in [Priority.critical, Priority.blocker]:
                add_in_change_list(ticket=self.key,
                                   text=f'I think this is **{priority_by_weight.capitalize()}**. Weight = {self.weight}')
            else:
                add_in_change_list(ticket=self.key,
                                   text=f'Change priority **{self.priority.key.capitalize()} -> {priority_by_weight.capitalize()}**.'
                                        f'Weight = {self.weight}')
                self.priority.key = priority_by_weight

        return prioritization


def processed_tickets(prioritization: Prioritization) -> None:
    if prioritization.is_list_ticket:
        issues = ticket
        prioritization = update_tickets(prioritization, issues)
    else:
        logging.info(f'Filter: {prioritization.filter}')
        issues = [item.key for item in client.issues.find(f'{prioritization.filter} AND Tags: !{PonyWorld.done}')]
        if not len(issues):
            logging.info("There aren't tickets.")
        while len(issues):
            prioritization = update_tickets(prioritization, issues)
            time.sleep(7)
            issues = [item.key for item in client.issues.find(f'{prioritization.filter} AND Tags: !{PonyWorld.done}')]

    logging.info("Delete tags...")
    delete_tag_pony(prioritization)
    logging.info("Tags was deleted")

    [logging.info(issue) for issue in prioritization.ticket_change_priority]
    prioritization.create_comments()


def update_tickets(prioritization: Prioritization, issues: list) -> Prioritization:
    for key in issues:
        item = client.issues[key]
        logging.info(f"Start processing for ticket {item.key}")
        data_item = Ticket(item, prioritization)

        if prioritization.is_update_audience and data_item.is_update_audience:
            audience = Audience()
            data_item.audience = audience.get_audience(data_item.event_name, data_item.platform)

        if prioritization.is_update_weight:
            data_item.weight = data_item.get_weight_for_ticket(data_item.audience)

        if prioritization.is_update_priotity:
            prioritization = data_item.update_priority(prioritization)

        logging.info(f"For ticket: Audience = {data_item.audience}, Weight = {data_item.weight}, "
                     f"Priority = {data_item.priority.key.capitalize()}, DuplicateCount = {data_item.duplicatesCount}, "
                     f"Type = {data_item.type}, Crash = {data_item.crash}, Reproduce = {data_item.reproduce}")

        update_fields = {'weightOne': data_item.weight, 'audience': data_item.audience, 'priority': data_item.priority}
        if not prioritization.is_list_ticket:
            update_fields['tags'] = {'add': [PonyWorld.done]}

        item.update(**update_fields)

        logging.info("The ticket {} was processed.\n\n".format(item.key))

    return prioritization


def delete_tag_pony(prioritization: Prioritization) -> None:
    issues = [item.key for item in client.issues.find(f'Queue: "Yandex Mobile Mail" AND Type: Bug AND Tags: inProd  AND (Tags: event_add OR Tags: event_skip ) AND Tags: {PonyWorld.done}')]

    while len(issues):
        for key in issues:
            new_revision = client.issues[key]
            new_revision.update(tags={'remove': [PonyWorld.done]}, ignore_version_change=True)
            logging.info(f"Tag {PonyWorld.done} delete for ticket {key}")
        time.sleep(7)
        issues = [item.key for item in client.issues.find(f'{prioritization.filter} AND Tags: {PonyWorld.done}')]


def main():
    parser = argparse.ArgumentParser(add_help=False)

    argument = parser.add_argument_group(title='Parametrs')

    argument.add_argument('--help', '-h', action='help', help='Helper')
    argument.add_argument('--not_priority', '-np',
                          action='store_const', const=True, default=False, help="Don't change priority")
    argument.add_argument('--not_weight', '-nw',
                          action='store_const', const=True, default=False, help="Don't change weight")
    argument.add_argument('--not_audience', '-na',
                          action='store_const', const=True, default=False, help="Don't change audience")
    argument.add_argument('--ticket', '-t',
                          action='store_const', const=True, default=False, help="Use massive of ticket")
    argument.add_argument('--filter_support', '-f',
                          action='store_const', const=True, default=False,
                          help="Filter for change field 'Duplicates Count'")
    argument.add_argument('--filter_only_new', '-fn',
                          action='store_const', const=True, default=False, help="Filter for change only new tickets")
    argument.add_argument('--filter_without_weight', '-iw',
                          action='store_const', const=True, default=False, help="Incorrect weight: weight empty or -1")

    flags = parser.parse_args()
    prioritization = Prioritization(flags)

    logging.info(f'Authorization data: User = {User.NAME},Token startrek = {User.TOKEN_ST},Token stat = {User.TOKEN_STAT}')

    logging.info(f"The script was run with argument:\n"
                 f"\t Update audience = {prioritization.is_update_audience}\n"
                 f"\t Update weight = {prioritization.is_update_weight}\n"
                 f"\t Update priority = {prioritization.is_update_priotity}\n"
                 f"\t Massive of tickets = {prioritization.is_list_ticket}\n"
                 f"\t Filter 'Change field Duplicates Count' = {flags.filter_support}\n"
                 f"\t Filter for only new tickets = {flags.filter_only_new}\n"
                 f"\t Filter for tickets with incorrect weight = {flags.filter_without_weight}\n")

    if flags.filter_support and flags.filter_only_new:
        logging.info(
            "The script was run with key '-f' and '-fn'. This is error. Restart the script with one of the keys")
        exit(0)

    audience = Audience()
    logging.info("Get all event from metrica....")
    prioritization.all_event = audience.get_all_event(platform='android')
    prioritization.all_event.extend(audience.get_all_event(platform='iOS'))
    prioritization.all_event = list(set(prioritization.all_event))
    prioritization.all_event.sort()
    logging.info("All event from metrica was gotten")

    processed_tickets(prioritization)


if __name__ == '__main__':
    main()
