import datetime
import os
import sys

import statface_client
import json
import logging
import argparse
from enum import Enum

from tp_api_client.tp_api_client import Action, TestPalmClient

sys.path.append(os.path.dirname(os.path.dirname(os.path.realpath(__file__))))
from reminder.constants import Team
from reminder.mail_sender import MailSender

logger = logging.getLogger('testpalm_logger')
logging.basicConfig(format=u'%(asctime)s [%(levelname)s] %(module)s: %(message)s',
                    level=logging.WARNING)
def handle_options():
    parser = argparse.ArgumentParser()
    parser.add_argument("-p", "--project", dest="project", help="Run on project iOS / Android")
    return parser

def get_project(args):
    if args.project and args.project in ['iOS', 'Android']:
        return args.project
    else:
        output('Project is not specified or project name incorrect (correct name: iOS or Android')
        sys.exit()


class User(Enum):
    TOKEN_STAT = os.environ['OAUTH_TOKEN_STAT']
    TOKEN_TP = os.environ['TESTPALM_TOKEN']
    NAME = 'username'
    APP = 'mail'


class FeaturePriority(Enum):
    HIGH = 'High'
    MED = 'Medium'
    LOW = 'Low'


class CaseImportance(Enum):
    HIGH = 'High'
    MED = 'Medium'
    LOW = 'Low'


class RegressLevels(Enum):
    ACC = 'Acceptance'
    BL = 'Business logic'
    REG = 'Regression'
    FULL = 'Full regression'
    ONCE = 'Once'


class Helpers(Enum):
    min_border_for_medium = float(1)
    max_border_for_medium = float(10)
    testpalm_project = 'mobilemail'

def output(string):
    print("{date}   {string}".format(date=datetime.datetime.now(), string=string))


class Audience:
    def __init__(self):
        self.client = statface_client.StatfaceClient(oauth_token=User.TOKEN_STAT.value,
                                                     host=statface_client.STATFACE_PRODUCTION)
        self.report = self.client.get_report('Mail/Totals/Mail_Mobile/EventAudience3')

    def get_all_event(self, platform: str) -> list:
        data = self.report.download_data(scale='daily',
                                         application=User.APP.value,
                                         _period_distance=30,
                                         device_type='_total_',
                                         os_major_version='_total_',
                                         platform=platform,
                                         session_type='_total_',
                                         type='json')

        output("Data from report was gotten.")

        if not len(data):
            output(
                "Data from report ia empty. All event for '{platform}' wasn't gotten.".format(platform=platform))
            return 0

        events = []
        for item in data:
            events.append(item['event_name'])

        output("Count of different event for iOS is {len}".format(platform=platform, len=len(events)))
        return events

    def get_audience_by_event(self, event: str, platform: str) -> float:
        data = self.report.download_data(scale='daily',
                                         application=User.APP.value,
                                         _period_distance=30,
                                         device_type='_total_',
                                         event_name=event,
                                         os_major_version='_total_',
                                         platform=platform,
                                         session_type='_total_',
                                         type='json')

        if not len(data):
            return 0

        average = 0
        for item in data:
            average += item['uuids_count']

        average = float(average / len(data))
        logging.info(f"Average count of '{event}' for '{platform}' == {average}.")
        return average

    def get_audience(self, event_name, platform):
        total = self.get_audience_by_event('_total_', platform)
        output(f"Total count events audience is {total}")
        event = self.get_audience_by_event(event_name, platform)
        event_audience = float(event / total)
        percent_audience = (round(event_audience * 100, 2))
        return percent_audience

    def count_priority(self, feature, platform):
        events = feature['events']
        product_weight = feature['product_weight']
        percent_audience = 0
        audience = Audience()
        for event in events:
            percent_unique_event_audience = audience.get_audience(event, platform)
            percent_audience += percent_unique_event_audience
        if percent_audience == 0:
            percent_audience = 0.01
        priority_percent = percent_audience * product_weight / 100
        output(f"Priority percent for feature {feature['feature']} is {priority_percent}")
        return priority_percent

    def get_priority_by_percent(self, priority_percent):
        if priority_percent < Helpers.min_border_for_medium.value:
            return FeaturePriority.LOW.value
        elif priority_percent < Helpers.max_border_for_medium.value:
            return FeaturePriority.MED.value
        else:
            return FeaturePriority.HIGH.value


class TestPalm:
    def __init__(self):
        self.client = TestPalmClient(auth=User.TOKEN_TP.value)

    def get_feature_testcases(self, feature, testpalm_project, feature_attribute_id):
        output(f"Getting testcases for feature {feature}")
        expression = '{{"type":"EQ","key":"attributes.{attribute_id}","value":"{feature}"}}'\
            .format(attribute_id=feature_attribute_id, feature=feature)

        return self.client.get_testcases(project=testpalm_project,
                                         include='id,attributes',
                                         expression=expression)

    def change_feature_priority_if_need(self, testpalm_project, case, feature_priority):
        self.client.update_testcases_attribute_value(project=testpalm_project,
                                                     case_ids=[case['id']],
                                                     action=Action.replace,
                                                     attr_name='Feature Priority',
                                                     attr_values=feature_priority)

    def change_case_priority(self, testpalm_project, case, feature_priority, case_importance_attribute_id):
        if case_importance_attribute_id not in case['attributes'].keys():
            output(f"Case with ID = {case['id']} doesn't have attribute with title Case Importance'.")
            return case['id']
        else:
            if len(case['attributes'][case_importance_attribute_id]) > 1:
                output(f'Case has more then one case importance attribute.')
                return case['id']
            else:
                case_importance = case['attributes'][case_importance_attribute_id][0]

                if feature_priority == FeaturePriority.HIGH.value and case_importance == CaseImportance.HIGH.value:
                    case_priority = RegressLevels.ACC.value
                elif (feature_priority == FeaturePriority.HIGH.value and case_importance == CaseImportance.MED.value) or (
                        feature_priority == FeaturePriority.MED.value and case_importance == CaseImportance.HIGH.value):
                    case_priority = RegressLevels.BL.value
                elif (feature_priority == FeaturePriority.HIGH.value and case_importance == CaseImportance.LOW.value) or (
                        feature_priority == FeaturePriority.MED.value and case_importance == CaseImportance.MED.value) or (
                        feature_priority == FeaturePriority.LOW.value and case_importance == CaseImportance.HIGH.value):
                    case_priority = RegressLevels.REG.value
                elif (feature_priority == FeaturePriority.MED.value and case_importance == CaseImportance.LOW.value) or (
                        feature_priority == FeaturePriority.LOW.value and case_importance == CaseImportance.MED.value):
                    case_priority = RegressLevels.FULL.value
                else:
                    case_priority = RegressLevels.ONCE.value
                self.client.update_testcases_attribute_value(project=testpalm_project,
                                                             case_ids=[case['id']],
                                                             action=Action.replace,
                                                             attr_name='Case Priority',
                                                             attr_values=[str(case_priority)])

def format_name_to_email(name):
    return name + '@yandex-team.ru'

def main():
    args = handle_options().parse_args()
    output('Authorization data: User = {}, Token stat = {}'.format(User.NAME, User.TOKEN_STAT))
    project = get_project(args)
    data = json.load(open(os.path.join(os.path.dirname(__file__), f'features_{project}.json')))
    features = data['features']
    audience = Audience()
    testpalm = TestPalm()
    feature_attribute_id = testpalm.client.get_attributes_id_by_title(Helpers.testpalm_project.value)[f'Feature {project}']
    case_importance_attribute_id = testpalm.client.get_attributes_id_by_title(Helpers.testpalm_project.value)['Case Importance']
    cases_without_case_importance = []
    output("Getting all event from metrica....")
    if project == 'Android':
        project = project.lower()
        recipients = Team.qa_android
    else:
        recipients = Team.qa_ios
    for i in range(len(recipients)):
        recipients[i] = format_name_to_email(recipients[i])
    all_event = audience.get_all_event(platform=project)
    all_event = list(set(all_event))
    all_event.sort()
    output(f"All event from metrica for platform {project} was gotten")

    for feature in features:
        priority_percent = audience.count_priority(feature, project)
        feature_priority = audience.get_priority_by_percent(priority_percent)
        output(f"Priority for feature {feature['feature']} is {feature_priority}")
        output(f"Feature is {feature['feature']}")
        cases = testpalm.get_feature_testcases(feature['feature'], Helpers.testpalm_project.value, feature_attribute_id)

        if not len(cases):
            output(f"There is no cases with Feature {feature['feature']}")
        for case in cases:
            testpalm.change_feature_priority_if_need(Helpers.testpalm_project.value, case, feature_priority)
            output(f"Field Feature Priority was changed for case {case['id']}")
            if testpalm.change_case_priority(Helpers.testpalm_project.value, case, feature_priority,
                                             case_importance_attribute_id) is not None:
                cases_without_case_importance.append(case['id'])
            else:
                output(f"Field Case Priority was changed for case {case['id']}")

    if len(cases_without_case_importance):
        for i in range(len(cases_without_case_importance)):
            cases_without_case_importance[i] = f'https://testpalm.yandex-team.ru/testcase/' \
                                               f'{Helpers.testpalm_project.value}-{cases_without_case_importance[i]}'
        subject = f'Неразмеченные кейсы в проекте {project}'
        header = 'Проблема с оценкой приоритета кейсов'
        body = f'В проекте {project} существуют неверно размеченные кейсы. ' \
               f'Case Importance отсутствует или имеет более одного значения в кейсах: {cases_without_case_importance}'
        message = MailSender(subject=subject, recipient_list=recipients, body=body, header=header)
        message.send()

if __name__ == '__main__':
    main()
