from datetime import datetime
import json
import os
import requests
import sys
import urllib3
import argparse
from urllib3 import Retry

from requests.adapters import HTTPAdapter
from startrek_client import Startrek
# from split_test_suite import get_and_run

import booking

urllib3.disable_warnings()

def connect():
    session = requests.Session()
    retry = Retry(total=5,
                  backoff_factor=0.3,
                  status_forcelist=(500, 502, 503, 504),
                  method_whitelist=frozenset(['GET', 'POST']))
    adapter = HTTPAdapter(max_retries=retry)
    session.mount('http://', adapter)
    session.mount('https://', adapter)
    return session


def handle_options():
    parser = argparse.ArgumentParser()
    parser.add_argument("-p", "--project", dest="project")
    parser.add_argument('-r', '--run_process', dest="process", action='store_const', const=True, default=False,
                        help="Run only process in Hitman")
    parser.add_argument('-b', '--booking', dest="booking",  action='store', nargs='?', type=str, const=None, default=False,
                        help="Booking for running")
    parser.add_argument('-s', '--split', dest="split", action='store_const', const=True, default=False,
                        help="Split big suite")
    return parser


def get_project(args):
    if args.project and args.project in ['iOS', 'Android']:
        return args.project
    else:
        output_in_console('Project is not specified or project name incorrect (correct name: iOS or Android)')
        sys.exit()


def output_in_console(string):
    print("{date}   {string}".format(date=datetime.now(), string=string))


def input_from_console(string):
    data = input("{date}   {string}".format(date=datetime.now(), string=string))
    return data


def stop_script(string):
    output_in_console('\x1b{}\x1b[0m'.format(string))
    sys.exit(0)


def check_return_code(request):
    status_OK = [200, 201]

    if not status_OK.count(request.status_code):
        output_in_console('\033[91m Error\033[00m in request. Status code = {}, reason = {}'
                          .format(request.status_code, request.reason))


def welcome_message(scenario):
    output_in_console('\x1b[1m\n'
                      'Hi!\n'
                      'Now we will create version and runs for Assessors'
                      'Script running with scenario {}\x1b[0m'.format(scenario))


def special_condition(data):
    text = '{text}\n' \
           'Обязательно в заведенных тикетах проставлять Найдено в версиях/Affected Version: ' \
           '{platform} {version} '.format(text='\n'.join(data['special_conditions']),
                                          platform=data['platform'],
                                          version=data['version_app'])

    check_condition = lambda text, condition, addition: '{}{}'.format(text, addition) if condition else text

    text = check_condition(text, data['part'], 'part {}'.format(data['part']))
    text = '{} testing\n'.format(text)
    text = check_condition(text, data['build_debug'],
                           'Ссылка на дебаг версию приложения: {}'.format(data['build_debug']))

    output_in_console('\x1b[1mInformation for assessors: \n\n {}\x1b[0m '.format(text))
    incorrect_data = input_from_console('\x1b[1m\n'
                                        'Is data correct? (tap \'Enter\' for continue; \'No\' for break) \x1b[0m ')

    if incorrect_data:
        stop_script('Running of script was stopped by user')

    return text


def run_process_in_hitman(args, text_for_assessors, data, version_testpalm, booking_id):
    output_in_console('Create task for assessors in Hitman....')

    project = get_project(args)
    process_code = 'mobmail_ios_auto' if project == 'iOS' else 'mobmail_android_auto'

    api_host = 'https://hitman.yandex-team.ru/'
    url = f"api/v1/execution/start/{process_code}"
    headers = {
        'Content-Type': 'application/json',
        'Authorization': 'OAuth ' + data['AUTH_HITMAN'],
    }

    body = {
        "requester": data['requester'],
        "properties": {
            "instruction": "https://wiki.eva.yandex-team.ru/search/assessors-testing/mobmail/",
            "namespace": str(data['testpalm_project']),
            "special_condition": text_for_assessors,
            "test_stend": data['build'],
            "parent_ticket": data['testing_ticket'],
            "is_report_write_to_ticket_required": 'true',
            "version": version_testpalm,
            "recommendation_time": int(75),
            "tickets_queue": data['queue'],
            "booking_id": booking_id
        }
    }

    response = requests.post(api_host + url, headers=headers, json=body, verify=False)
    print(response.text)


def give_task_to_assessors(data, text_for_assessors, tp=None):
    output_in_console('Check version running (Hitman with booking)....')

    args = handle_options().parse_args()
    version_testpalm = tp.version_testpalm if tp \
        else 'Assessors{platform}{version}part{part}'.format(platform=data['platform'],
                                                             version=str(data['version_app']).replace('.', ''),
                                                             part=str(data['part']))

    output_in_console("Start with booking")
    project = get_project(args)
    if args.booking:
        booking_id = args.booking
    else:
        book = booking.booking_version_testpalm(project, version_testpalm)
        booking_date = datetime.fromtimestamp(int(book['estimate']['startTs']/1000))
        booking_id = book['bookingId']
        output_in_console('\x1b[1mBooking create on: date={}.{}, time={}.{}\x1b[0m'.format(booking_date.day,
                                                                                           booking_date.month,
                                                                                           booking_date.hour,
                                                                                           booking_date.minute))

    output_in_console("Start with Hitman. Id bron is {}".format(booking_id))
    run_process_in_hitman(args, text_for_assessors, data, version_testpalm, booking_id=booking_id)

    output_in_console('The task for assessors was given!')


class StartTrack:
    def __init__(self, data):
        self.client = Startrek(base_url='https://st-api.yandex-team.ru',
                               useragent=data['requester'],
                               token=data['AUTH_ST'])

        self.requester = data['requester']
        self.ticket_version = data['testing_ticket']
        self.platform = str(data['platform'])
        self.queue = data['queue']

        self.version = \
            '{} part {} testing'.format(str(data['version_app']), str(data['part'])) if data['part'] else str(
                data['version_app'])

        self.id_version = False
        self.task_key = False

    def create_version_startrek(self):
        version = self.client.versions.create(queue=self.queue, name='{} {}'.format(self.platform, self.version))
        self.id_version = version.id
        output_in_console('Version in StarTrack was created.')

    def create_ticket_for_testing(self):
        ticket = self.client.issues.create(queue=self.queue,
                                           summary='{} {}'.format(self.platform, self.version),
                                           type='task',
                                           affectedVersions=self.id_version,
                                           fixVersions=self.id_version,
                                           assignee=self.requester)
        self.task_key = ticket.key
        output_in_console("Ticket for testing was created in Startrek: \x1b[1m {} \x1b[0m".format(self.task_key))
        self.create_comment()

    def create_comment(self):
        testing_ticket = self.client.issues[self.ticket_version]
        testing_ticket.comments.create(text='Create ticket for release notes: {}'.format(self.task_key))
        output_in_console("Comment in Ticket for testing of version was created in Startrek ")


class TestPalm:
    def __init__(self, data, ticket_for_testing, id_version, suites):
        self.ticket_for_testing = ticket_for_testing
        self.id_version = id_version

        self.version_app = str(data['version_app']).replace('.', '')
        if data['part']:
            self.version_app = '{version}part{part}'.format(version=self.version_app,
                                                            part=str(data['part']))

        self.testpalm_project = data['testpalm_project']
        self.queue = data['queue']
        self.version_testpalm = 'Assessors{}{}'.format(data['platform'], self.version_app)
        self.suites = suites

        self.headers_testpalm = {
            'Authorization': 'OAuth ' + data['AUTH_TP'],
            'Content-Type': 'application/json'}

    def create_run_for_assessors(self):
        output_in_console("Create runs....")

        for item in self.suites:
            self.create_testrun(item, '{version} {name} {enviroment}'.format(version=self.version_app,
                                                                             name=item['run'],
                                                                             enviroment=item['enviroment'].strip()))

            output_in_console('Run \'{name} {enviroment}\' was created'.format(name=item['run'],
                                                                               enviroment=item['enviroment']))

    def create_testpalm_version(self, id_version):
        data = {
            'trackerVersion': {
                'groupId': 'MOBILEMAIL',
                'isClosed': False,
                'title': self.version_testpalm,
                'trackerId': 'Startrek',
                'versionId': str(id_version),
                'url': 'https://st.yandex-team.ru/MOBILEMAIL/filter?fixVersions={}'.format(id_version)
            },
            'id': self.version_testpalm
        }

        rawBody = json.dumps(data, ensure_ascii=False, separators=(',', ': '))
        rawBody = rawBody.encode('utf-8')

        r = connect().post('https://testpalm.yandex-team.ru:443/api/version/{}'.format(self.testpalm_project),
                          data=rawBody,
                          headers=self.headers_testpalm)
        check_return_code(r)

        output_in_console("Version {} in Testpalm  was created".format(self.version_testpalm))

    def create_testrun(self, suite, title):
        data = {
            'title': title,
            'version': self.version_testpalm,
            'currentEnvironment': {
                'title': suite['enviroment'],
                'description': suite['enviroment'],
                'default': False
            },
            'tags': [suite['tags']],
            'parentIssue': {
                'assignee': [],
                'createdTime': 0,
                'display': False,
                'groupId': self.queue,
                'id': False,
                'isBug': False,
                'isResolved': False,
                'title': self.queue,
                'trackerId': 'Startrek'
            }
        }
        print (f"data {data}")

        rawBody = json.dumps(data, ensure_ascii=False, separators=(',', ': '))
        rawBody = rawBody.encode('utf-8')

        r = connect().post(
            'https://testpalm.yandex-team.ru:443/api/testrun/{}/testsuite/{}'.format(self.testpalm_project,
                                                                                     str(suite['key'])),
            data=rawBody,
            headers=self.headers_testpalm)

        check_return_code(r)


def main():
    args = handle_options().parse_args()
    project = get_project(args)
    testing_data = json.load(open(os.path.join(os.path.dirname(__file__), f'testing_data_{project}.json')))
    suites = json.load(open(os.path.join(os.path.dirname(__file__), f'suites_{project}.json')))

    scenario = testing_data['scenario']
    welcome_message(scenario)
    text_for_assessors = special_condition(testing_data)

    incorrect_data = input_from_console('\x1b[1m\n'
                                        'Link is build: {}\n'
                                        'Is it correct? (tap \'Enter\' for continue; \'No\' for break) \x1b[0m '
                                        .format(testing_data['build']))

    if incorrect_data:
        stop_script('Running of script was stopped by user')

    st = StartTrack(testing_data)

    if args.process:
        give_task_to_assessors(testing_data, text_for_assessors)
        stop_script('\x1b[1mScript finished successfully!\x1b[0m')

    st.create_version_startrek()
    st.create_ticket_for_testing()
    tp = TestPalm(testing_data,
                  st.task_key,
                  st.id_version,
                  suites[scenario])
    if args.split:
        get_and_run(testing_data, suites, st.task_key)
    else:
        tp.create_testpalm_version(id_version=st.id_version)
        tp.create_run_for_assessors()

    give_task_to_assessors(testing_data, text_for_assessors, tp)

    output_in_console('\x1b[1mScript finished successfully!\x1b[0m')


if __name__ == '__main__':
    main()
